<?php
// Start the session
session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    $_SESSION['error'] = "You must be logged in to perform this action";
    header("Location: login.php");
    exit();
}

// Check if transaction data exists
if (!isset($_SESSION['transaction_data'])) {
    $_SESSION['error'] = "No transaction data found";
    header("Location: dashboard.php");
    exit();
}

// Check if transaction data has expired
if ($_SESSION['transaction_data']['expires'] < time()) {
    unset($_SESSION['transaction_data']);
    $_SESSION['error'] = "Transaction data has expired";
    header("Location: dashboard.php");
    exit();
}

// Get transaction data
$tx_data = $_SESSION['transaction_data'];

// Cryptocurrency symbols and other data
$crypto_symbols = [
    'btc' => 'BTC',
    'eth' => 'ETH',
    'usdt' => 'USDT',
    'sol' => 'SOL',
    'ada' => 'ADA',
    'xrp' => 'XRP',
    'doge' => 'DOGE',
    'bnb' => 'BNB',
    'pepe' => 'PEPE'
];

// Get crypto symbol
$crypto_symbol = isset($crypto_symbols[$tx_data['crypto_type']]) ? 
                $crypto_symbols[$tx_data['crypto_type']] : 
                strtoupper($tx_data['crypto_type']);

// Network labels
$network_labels = [
    'btc' => 'Bitcoin Network',
    'eth' => 'Ethereum Network',
    'bsc' => 'Binance Smart Chain',
    'polygon' => 'Polygon Network',
    'trc20' => 'Tron Network',
    'sol' => 'Solana Network',
    'ada' => 'Cardano Network'
];

// Get network name
$network_name = isset($network_labels[$tx_data['network']]) ? 
               $network_labels[$tx_data['network']] : 
               ucfirst($tx_data['network']) . ' Network';

// Process the transaction if confirmed
if (isset($_POST['confirm'])) {
    // Add confirmation flag
    $_POST['confirmed'] = 'true';
    
    // Redirect to processor with all POST data
    echo '<form id="confirmForm" action="process_send.php" method="post">';
    
    // Add all transaction data as hidden inputs
    foreach ($tx_data as $key => $value) {
        if ($key !== 'expires') {
            echo '<input type="hidden" name="' . htmlspecialchars($key) . '" value="' . htmlspecialchars($value) . '">';
        }
    }
    
    // Add confirmation flag
    echo '<input type="hidden" name="confirmed" value="true">';
    
    echo '</form>';
    echo '<script>document.getElementById("confirmForm").submit();</script>';
    exit();
}

// Page title
$page_title = "Confirm Your Transaction";

// Include header
include('includes/header.php');
?>

<div class="container mx-auto px-4 py-8">
    <!-- Back Button -->
    <div class="mb-6">
        <a href="dashboard.php" class="flex items-center text-gray-400 hover:text-primary-500">
            <i class="fas fa-arrow-left mr-2"></i> Back to Dashboard
        </a>
    </div>
    
    <!-- Confirmation Card -->
    <div class="bg-dark-900 rounded-xl p-6 shadow-lg max-w-2xl mx-auto">
        <h2 class="text-2xl font-bold mb-6 text-center">Confirm Your Transaction</h2>
        
        <!-- Warning Message -->
        <div class="bg-yellow-900 bg-opacity-20 border border-yellow-700 rounded-lg p-4 mb-6">
            <div class="flex items-start">
                <div class="bg-yellow-600 rounded-full p-2 mr-3 shrink-0">
                    <i class="fas fa-exclamation-triangle text-dark-900"></i>
                </div>
                <div>
                    <h3 class="font-bold text-yellow-500">Important</h3>
                    <p class="text-sm">Please verify all transaction details carefully. Cryptocurrency transactions are irreversible and cannot be refunded.</p>
                </div>
            </div>
        </div>
        
        <!-- Transaction Details -->
        <div class="bg-dark-800 rounded-xl p-5 mb-6">
            <h3 class="font-bold mb-4 text-lg">Transaction Details</h3>
            
            <div class="space-y-3">
                <!-- Amount -->
                <div class="flex justify-between items-center">
                    <span class="text-gray-400">Sending Amount:</span>
                    <span class="font-bold"><?php echo number_format($tx_data['crypto_amount'], 8); ?> <?php echo $crypto_symbol; ?></span>
                </div>
                
                <!-- Network Fee -->
                <div class="flex justify-between items-center">
                    <span class="text-gray-400">Network Fee:</span>
                    <span><?php echo number_format($tx_data['fee'], 8); ?> <?php echo $crypto_symbol; ?></span>
                </div>
                
                <!-- Total Amount -->
                <div class="flex justify-between items-center border-t border-dark-700 pt-3 mt-3">
                    <span class="text-gray-400">Total:</span>
                    <span class="font-bold"><?php echo number_format($tx_data['total_amount'], 8); ?> <?php echo $crypto_symbol; ?></span>
                </div>
                
                <?php if(isset($tx_data['usd_amount']) && $tx_data['usd_amount'] > 0): ?>
                <div class="flex justify-between items-center text-sm text-gray-500">
                    <span>USD Value:</span>
                    <span>$<?php echo number_format($tx_data['usd_amount'], 2); ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Recipient Details -->
        <div class="bg-dark-800 rounded-xl p-5 mb-6">
            <h3 class="font-bold mb-4 text-lg">Recipient Details</h3>
            
            <div class="space-y-3">
                <!-- Wallet Address -->
                <div>
                    <span class="text-gray-400 block mb-1">Wallet Address:</span>
                    <div class="bg-dark-900 p-3 rounded-lg break-all font-mono text-sm"><?php echo $tx_data['wallet']; ?></div>
                </div>
                
                <!-- Network -->
                <div class="flex justify-between items-center">
                    <span class="text-gray-400">Network:</span>
                    <span><?php echo $network_name; ?></span>
                </div>
                
                <?php if(!empty($tx_data['name'])): ?>
                <!-- Recipient Name (if provided) -->
                <div class="flex justify-between items-center">
                    <span class="text-gray-400">Recipient Name:</span>
                    <span><?php echo htmlspecialchars($tx_data['name']); ?></span>
                </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Action Buttons -->
        <div class="flex flex-col space-y-3 sm:flex-row sm:space-y-0 sm:space-x-3">
            <a href="dashboard.php" class="flex-1 bg-dark-700 hover:bg-dark-600 text-center py-4 rounded-xl font-bold">Cancel</a>
            
            <form method="post" class="flex-1">
                <button type="submit" name="confirm" class="w-full bg-primary-600 hover:bg-primary-700 text-white py-4 rounded-xl font-bold">
                    Confirm and Send
                </button>
            </form>
        </div>
    </div>
    
    <!-- Disclaimer -->
    <div class="text-center mt-6 text-sm text-gray-500 max-w-2xl mx-auto">
        <p>By confirming this transaction, you acknowledge that you are sending cryptocurrency to the specified address. Make sure you have verified the recipient's address before proceeding.</p>
    </div>
</div>

<?php
// Include footer
include('includes/footer.php');
?>