<?php
// Enable error reporting (for development only)
// Remove or disable in production
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in; if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$usd_balance    = isset($userDetails['balance']) ? (float)$userDetails['balance'] : 0.00;
$btc            = isset($userDetails['btc']) ? (float)$userDetails['btc'] : 0.000000;
$eth            = isset($userDetails['eth']) ? (float)$userDetails['eth'] : 0.000000;
$usdt           = isset($userDetails['usdt']) ? (float)$userDetails['usdt'] : 0.000000;
$xrp            = isset($userDetails['xrp']) ? (float)$userDetails['xrp'] : 0.000000;
$doge           = isset($userDetails['doge']) ? (float)$userDetails['doge'] : 0.000000;
$ada            = isset($userDetails['ada']) ? (float)$userDetails['ada'] : 0.000000;
$extra          = $userDetails['extra'];
$update_time    = $userDetails['update_time'];

// Define cache file and cache duration (2 hours)
$cache_file = "ada_price_cache.json";
$cache_duration = 7200; // 2 hours

// Function to fetch ADA price from CoinGecko API using cURL
function fetch_ada_price() {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=cardano&vs_currencies=usd";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Ignore SSL verification issues
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Timeout after 10 seconds
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Ensure response is valid and HTTP status code is 200 (OK)
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return false;
}

// Use cache if valid
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_duration) {
    $data = json_decode(file_get_contents($cache_file), true);
} else {
    // Fetch fresh ADA price
    $data = fetch_ada_price();

    // Store new price in cache if valid
    if ($data && isset($data['cardano']['usd'])) {
        file_put_contents($cache_file, json_encode($data));
    }
}

// Ensure ADA price is available
$ada_price = $data['cardano']['usd'] ?? null;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cardano (ADA) | primemarketspro.com</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <link rel="icon" href="https://primemarketspro.com/session/favicon.png" type="image/x-icon">
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <style>
        .ada-bg { background-color: #0033AD; }
        .ada-text { color: #0033AD; }
        .ada-secondary { color: #3468DC; }
        .btn-ada {
            background: linear-gradient(135deg, #0033AD 0%, #3468DC 100%);
            color: white;
            transition: all 0.3s;
            border: none;
        }
        .btn-ada:hover {
            background: linear-gradient(135deg, #002A91 0%, #0033AD 100%);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(0, 51, 173, 0.3);
        }
        .tradingview-widget-container {
            width: 100%;
            min-height: 450px;
        }
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .progress-bar {
            height: 4px;
            background-color: #e2e8f0;
            border-radius: 2px;
            overflow: hidden;
        }
        .progress-value {
            height: 100%;
            background: linear-gradient(90deg, #0033AD 0%, #3468DC 100%);
            transition: width 0.5s ease;
        }
        .description-collapse {
            max-height: 100px;
            overflow: hidden;
            position: relative;
        }
        .description-collapse::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(to top, rgba(255,255,255,1), rgba(255,255,255,0));
        }
        .description-expanded {
            max-height: none;
        }
        .research-indicator {
            background: linear-gradient(135deg, #0033AD 0%, #3468DC 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
            margin-top: 4px;
        }
        .ada-card {
            background: linear-gradient(135deg, #F0F4FF 0%, #E6EFFF 100%);
            border: 2px solid #3468DC;
        }
        .price-bounce {
            animation: bounce 2s infinite;
        }
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }
        .feature-tag {
            background: linear-gradient(135deg, #0033AD 0%, #3468DC 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .green-tag {
            background: linear-gradient(135deg, #10B981 0%, #059669 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .staking-indicator {
            background: linear-gradient(135deg, #7C3AED 0%, #5B21B6 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .scientific-text {
            font-family: 'Georgia', serif;
            color: #0033AD;
            font-style: italic;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen p-4">
    <div class="max-w-4xl mx-auto bg-white p-4 md:p-6 rounded-2xl shadow-sm">
        <!-- Header -->
        <header class="flex items-center justify-between mb-8">
            <button onclick="history.back()" class="ada-text p-2 rounded-full hover:bg-gray-100 transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                </svg>
            </button>
            <h1 class="text-xl font-semibold text-gray-800 flex items-center">
                <img src="https://primemarketspro.com/cryptos/cardano.png" class="w-8 h-8 mr-2" alt="Cardano">
                Cardano (ADA)
            </h1>
            <div class="p-2 rounded-full hover:bg-gray-100 transition-colors">
                <div id="connection-status" class="w-3 h-3 rounded-full bg-green-500"></div>
            </div>
        </header>

        <!-- Price Section -->
        <section class="text-center mb-8 fade-in">
            <div class="price-text text-4xl font-bold mb-2 flex items-center justify-center">
                <span id="ada-price" class="text-gray-800">$0.00</span>
                <span id="ada-arrow" class="ml-2"></span>
            </div>
            <div id="ada-change" class="text-lg font-medium text-gray-600">0.00% (24h)</div>
            <div class="research-indicator">
                🔬 Peer-Reviewed Blockchain Research
            </div>
            <div class="text-sm font-medium text-blue-500 mt-2">
                1 USD ≈ <span id="ada-equivalent" class="text-gray-700">0.00 ADA</span>
            </div>
            <div class="flex justify-center mt-2 space-x-1">
                <span class="feature-tag">PoS</span>
                <span class="green-tag">♻️ Sustainable</span>
                <span class="staking-tag">💎 Staking</span>
                <span class="feature-tag">Smart Contracts</span>
            </div>
        </section>

       
      <!-- TradingView Widget Section -->
<section class="mb-6" style="width: 100%;">
    <div class="tradingview-widget-container" style="width: 100%; height: 420px;">
        <div class="tradingview-widget-container__widget" style="width: 100%; height: 100%;"></div>
        <script type="text/javascript" src="https://s3.tradingview.com/external-embedding/embed-widget-mini-symbol-overview.js" async>
        {
            "symbol": "BINANCE:ADAUSDT",
            "width": "100%",
            "height": "250",
            "locale": "en",
            "dateRange": "12M",
            "colorTheme": "light",
            "isTransparent": true,
            "autosize": false,
            "largeChartUrl": ""
        }
        </script>
    </div>
</section>


        <!-- Wallet Card -->
        <section class="wallet-card ada-card rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm">Your ADA Balance</p>
                    <h2 class="text-2xl font-bold text-gray-800 mt-1">
                        <?php echo number_format($ada, 2); ?> ADA
                    </h2>
                    <p class="text-gray-600 mt-1">
                        ≈ $<span id="ada-balance-usd"><?php echo number_format($ada * 1.00, 2); // Initial estimate ?></span> USD
                    </p>
                    <p class="scientific-text text-xs mt-1">Scientifically designed for the future</p>
                </div>
                <img src="https://primemarketspro.com/cryptos/cardano.png" class="w-12 h-12" alt="Cardano">
            </div>
            <div class="mt-4 flex space-x-2">
                <a href="adas.php" class="btn-ada py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-up mr-2"></i>Send
                </a>
                <a href="adar.php" class="btn-ada py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-down mr-2"></i>Receive
                </a>
            </div>
        </section>

        <!-- Market Data -->
        <section class="bg-gray-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Market Overview</h3>
            <div class="space-y-4 text-sm">
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap Rank</span>
                        <span id="market-cap-rank">#8</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap</span>
                        <span id="market-cap">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>24h Volume</span>
                        <span id="volume">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Circulating Supply</span>
                        <span id="supply">0 ADA</span>
                    </div>
                    <div class="progress-bar mt-1">
                        <div id="supply-progress" class="progress-value" style="width: 0%"></div>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        <span id="circulating-supply">0</span> / <span id="total-supply">45B</span> ADA
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Staking Participation</span>
                        <span id="staking-rate" class="text-purple-600 font-medium">~73%</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Percentage of ADA actively staked
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Block Time</span>
                        <span id="block-time" class="text-green-600 font-medium">~20 seconds</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Average epoch slot time
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Energy Efficiency</span>
                        <span id="energy-efficiency" class="text-green-600 font-medium">99.95% less than Bitcoin</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Proof-of-Stake consensus mechanism
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Smart Contracts</span>
                        <span id="smart-contracts" class="text-blue-600 font-medium">Plutus Active</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Haskell-based smart contract platform
                    </div>
                </div>
            </div>
        </section>

        <!-- Research & Sustainability Info -->
        <section class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-blue-600 text-lg">🔬</span>
                </div>
                <div>
                    <h4 class="font-semibold text-blue-800 mb-1">Third-Generation Blockchain</h4>
                    <p class="text-sm text-blue-700">
                        Cardano is built on peer-reviewed research and evidence-based methods. It aims to solve scalability, interoperability, and sustainability challenges through a layered architecture and rigorous academic approach.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-blue-600">✓ Peer-reviewed research foundation</div>
                        <div class="text-xs text-blue-600">✓ 99.95% more energy efficient than Bitcoin</div>
                        <div class="text-xs text-blue-600">✓ Ouroboros Proof-of-Stake consensus</div>
                        <div class="text-xs text-blue-600">✓ Plutus smart contract platform</div>
                        <div class="text-xs text-blue-600">✓ Native tokens and NFTs</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- Staking Info -->
        <section class="bg-purple-50 border border-purple-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-purple-600 text-lg">💎</span>
                </div>
                <div>
                    <h4 class="font-semibold text-purple-800 mb-1">Liquid Staking Rewards</h4>
                    <p class="text-sm text-purple-700">
                        Earn rewards by staking your ADA while maintaining full liquidity. No lock-up periods, no slashing, and delegate to multiple pools for optimal rewards and decentralization.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-purple-600">✓ 4-5% annual staking rewards</div>
                        <div class="text-xs text-purple-600">✓ No lock-up or minimum requirements</div>
                        <div class="text-xs text-purple-600">✓ Delegate to multiple stake pools</div>
                        <div class="text-xs text-purple-600">✓ No slashing penalties</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- About Section -->
        <section class="mb-8 fade-in">
            <div class="flex justify-between items-center mb-4">
                <h3 class="font-semibold text-gray-800">About Cardano</h3>
                <button id="toggle-description" class="text-blue-500 text-sm font-medium">Show More</button>
            </div>
            <div id="description-content" class="description-collapse text-gray-600 text-sm leading-relaxed">
                Loading Cardano description...
            </div>
        </section>

        <!-- Official Links -->
        <section class="fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Resources</h3>
            <div class="grid grid-cols-2 gap-3">
                <a href="https://cardano.org" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Official Website</div>
                </a>
                <a href="https://github.com/input-output-hk/cardano-node" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Source Code</div>
                </a>
                <a href="blockchain_details.php" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Block Explorer</div>
                </a>
                <a href="https://iohk.io/en/research/library/" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Research Papers</div>
                </a>
                <a href="https://forum.cardano.org" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Community Forum</div>
                </a>
                <a href="https://docs.cardano.org" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Documentation</div>
                </a>
            </div>
        </section>
    </div>

    <script>
        // DOM Elements
        const adaPriceEl = document.getElementById('ada-price');
        const adaArrowEl = document.getElementById('ada-arrow');
        const adaChangeEl = document.getElementById('ada-change');
        const adaEquivalentEl = document.getElementById('ada-equivalent');
        const adaBalanceUsdEl = document.getElementById('ada-balance-usd');
        const marketCapEl = document.getElementById('market-cap');
        const volumeEl = document.getElementById('volume');
        const supplyEl = document.getElementById('supply');
        const marketCapRankEl = document.getElementById('market-cap-rank');
        const stakingRateEl = document.getElementById('staking-rate');
        const blockTimeEl = document.getElementById('block-time');
        const energyEfficiencyEl = document.getElementById('energy-efficiency');
        const smartContractsEl = document.getElementById('smart-contracts');
        const descriptionContentEl = document.getElementById('description-content');
        const toggleDescriptionBtn = document.getElementById('toggle-description');
        const connectionStatusEl = document.getElementById('connection-status');
        const supplyProgressEl = document.getElementById('supply-progress');
        const circulatingSupplyEl = document.getElementById('circulating-supply');
        const totalSupplyEl = document.getElementById('total-supply');

        // User data from PHP
        const userADA = <?php echo json_encode($ada); ?>;
        let previousPrice = 0;
        let coinData = {};
        let descriptionExpanded = false;

        // Toggle description
        toggleDescriptionBtn.addEventListener('click', () => {
            descriptionExpanded = !descriptionExpanded;
            descriptionContentEl.classList.toggle('description-collapse', !descriptionExpanded);
            descriptionContentEl.classList.toggle('description-expanded', descriptionExpanded);
            toggleDescriptionBtn.textContent = descriptionExpanded ? 'Show Less' : 'Show More';
        });

        // Format numbers with commas
        function formatNumber(num, decimals = 0) {
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(num);
        }

        // Format currency
        function formatCurrency(amount) {
            return new Intl.NumberFormat('en-US', {
                style: 'currency',
                currency: 'USD',
                minimumFractionDigits: 3,
                maximumFractionDigits: 4
            }).format(amount);
        }

        // Update UI with price data
        function updatePriceUI(price, change) {
            // Update price
            adaPriceEl.textContent = formatCurrency(price);
            
            // Update 24h change
            const changeFixed = change.toFixed(2);
            adaChangeEl.textContent = `${changeFixed}% (24h)`;
            adaChangeEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update arrow
            adaArrowEl.textContent = change >= 0 ? '↑' : '↓';
            adaArrowEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update equivalent
            const adaEquivalent = 1 / price;
            adaEquivalentEl.textContent = adaEquivalent.toFixed(2) + ' ADA';
            
            // Update user balance
            const adaBalanceUSD = userADA * price;
            adaBalanceUsdEl.textContent = formatNumber(adaBalanceUSD, 2);
            
            // Price change animation with bounce effect
            if (previousPrice && previousPrice !== price) {
                adaPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                void adaPriceEl.offsetWidth; // Trigger reflow
                adaPriceEl.classList.add(price > previousPrice ? 'text-green-500' : 'text-red-500');
                adaPriceEl.classList.add('price-bounce');
                
                setTimeout(() => {
                    adaPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                    adaPriceEl.classList.add('text-gray-800');
                }, 3000);
            }
            
            previousPrice = price;
        }

        // Update market data
        function updateMarketData(data) {
            marketCapEl.textContent = formatCurrency(data.market_data.market_cap.usd);
            volumeEl.textContent = formatCurrency(data.market_data.total_volume.usd);
            supplyEl.textContent = formatNumber(data.market_data.circulating_supply / 1000000000, 1) + 'B ADA';
            marketCapRankEl.textContent = '#' + data.market_cap_rank;
            
            // Update supply progress (Cardano has max supply of 45B)
            const circulating = data.market_data.circulating_supply;
            const maxSupply = 45000000000; // 45 billion ADA max supply
            const supplyPercentage = (circulating / maxSupply) * 100;
            supplyProgressEl.style.width = supplyPercentage + '%';
            circulatingSupplyEl.textContent = formatNumber(circulating / 1000000000, 1) + 'B';
            
            // Update staking rate based on market activity
            const volume = data.market_data.total_volume.usd;
            const marketCap = data.market_data.market_cap.usd;
            const volumeRatio = volume / marketCap;
            
            if (volumeRatio > 0.1) {
                stakingRateEl.textContent = '~71% Active';
                stakingRateEl.className = 'text-green-600 font-medium';
            } else if (volumeRatio > 0.05) {
                stakingRateEl.textContent = '~73% Staked';
                stakingRateEl.className = 'text-purple-600 font-medium';
            } else {
                stakingRateEl.textContent = '~75% Staked';
                stakingRateEl.className = 'text-blue-600 font-medium';
            }
            
            // Update smart contracts status based on network activity
            if (data.market_data.price_change_percentage_24h > 5) {
                smartContractsEl.textContent = 'Plutus Very Active';
                smartContractsEl.className = 'text-green-600 font-medium';
            } else if (data.market_data.price_change_percentage_24h > 0) {
                smartContractsEl.textContent = 'Plutus Active';
                smartContractsEl.className = 'text-blue-600 font-medium';
            } else {
                smartContractsEl.textContent = 'Plutus Stable';
                smartContractsEl.className = 'text-gray-600 font-medium';
            }
        }

        // Fetch ADA data from CoinGecko
        async function fetchADAData() {
            try {
                const response = await axios.get('https://api.coingecko.com/api/v3/coins/cardano', {
                    params: {
                        localization: false,
                        tickers: false,
                        community_data: false,
                        developer_data: false,
                        sparkline: false
                    },
                    timeout: 10000
                });
                
                coinData = response.data;
                const price = coinData.market_data.current_price.usd;
                const change = coinData.market_data.price_change_percentage_24h;
                updatePriceUI(price, change);
                updateMarketData(coinData);
                
                // Update description
                if (coinData.description && coinData.description.en) {
                    descriptionContentEl.innerHTML = coinData.description.en;
                } else {
                    descriptionContentEl.innerHTML = `
                        <p>Cardano is a third-generation blockchain platform that aims to provide a more sustainable and scalable infrastructure for the development of decentralized applications and smart contracts.</p>
                        <p>Built on peer-reviewed research and evidence-based methods, Cardano uses the Ouroboros proof-of-stake consensus mechanism, which is significantly more energy-efficient than traditional proof-of-work systems.</p>
                        <p>The platform features a layered architecture that separates the settlement layer (Cardano Settlement Layer) from the computation layer (Cardano Computation Layer), allowing for greater flexibility and upgradability.</p>
                        <p>Cardano's native smart contract platform, Plutus, is built using Haskell, a functional programming language known for its mathematical rigor and security features.</p>
                    `;
                }
                
                // Update connection status
                connectionStatusEl.className = 'w-3 h-3 rounded-full bg-green-500';
                connectionStatusEl.title = 'Connected to CoinGecko API';
                
            } catch (error) {
                console.error('Error fetching ADA data:', error);
                
                // Fallback to cached price if available
                <?php if ($ada_price): ?>
                const cachedPrice = <?php echo json_encode($ada_price); ?>;
                updatePriceUI(cachedPrice, 0);
                <?php endif; ?>
                
                // Update connection status to indicate error
                connectionStatusEl.className = 'w-3 h-3 rounded-full bg-red-500';
                connectionStatusEl.title = 'Connection error - using cached data';
                
                // Set fallback description
                descriptionContentEl.innerHTML = `
                    <p>Cardano is a third-generation blockchain platform that aims to provide a more sustainable and scalable infrastructure for the development of decentralized applications and smart contracts.</p>
                    <p>Built on peer-reviewed research and evidence-based methods, Cardano uses the Ouroboros proof-of-stake consensus mechanism, which is significantly more energy-efficient than traditional proof-of-work systems.</p>
                    <p>The platform features a layered architecture that separates the settlement layer from the computation layer, allowing for greater flexibility and upgradability.</p>
                `;
            }
        }

        // Real-time price updates
        function startRealTimeUpdates() {
            // Initial fetch
            fetchADAData();
            
            // Update every 30 seconds
            setInterval(fetchADAData, 30000);
            
            // Update connection status indicator
            setInterval(() => {
                const currentTime = new Date().getTime();
                const lastUpdate = parseInt(localStorage.getItem('lastADAUpdate') || '0');
                
                if (currentTime - lastUpdate > 60000) { // More than 1 minute
                    connectionStatusEl.className = 'w-3 h-3 rounded-full bg-yellow-500';
                    connectionStatusEl.title = 'Slow connection';
                } else {
                    connectionStatusEl.className = 'w-3 h-3 rounded-full bg-green-500';
                    connectionStatusEl.title = 'Connected';
                }
            }, 10000);
        }

        // Start updates when page loads
        document.addEventListener('DOMContentLoaded', startRealTimeUpdates);

        // Handle page visibility changes
        document.addEventListener('visibilitychange', () => {
            if (document.visibilityState === 'visible') {
                fetchADAData(); // Refresh data when page becomes visible
            }
        });

        // Add smooth scrolling for internal links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', function (e) {
                e.preventDefault();
                const target = document.querySelector(this.getAttribute('href'));
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });

        // Add loading states for buttons
        document.querySelectorAll('.btn-ada').forEach(button => {
            button.addEventListener('click', function(e) {
                if (this.classList.contains('loading')) {
                    e.preventDefault();
                    return;
                }
                
                this.classList.add('loading');
                const originalText = this.innerHTML;
                this.innerHTML = '<span class="inline-block animate-spin mr-2">⟳</span>Loading...';
                
                // Remove loading state after navigation
                setTimeout(() => {
                    this.classList.remove('loading');
                    this.innerHTML = originalText;
                }, 2000);
            });
        });

        // Enhanced error handling with retry mechanism
        let retryCount = 0;
        const maxRetries = 3;

        function fetchADADataWithRetry() {
            fetchADAData().catch(error => {
                retryCount++;
                if (retryCount < maxRetries) {
                    console.log(`Retrying... (${retryCount}/${maxRetries})`);
                    setTimeout(fetchADADataWithRetry, 5000 * retryCount); // Exponential backoff
                } else {
                    console.error('Max retries reached. Using fallback data.');
                    retryCount = 0;
                }
            });
        }

        // Initialize tooltips for better UX
        const tooltipElements = document.querySelectorAll('[title]');
        tooltipElements.forEach(element => {
            element.addEventListener('mouseenter', function() {
                this.style.position = 'relative';
            });
        });

        // Add keyboard navigation support
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && descriptionExpanded) {
                toggleDescriptionBtn.click();
            }
        });

        // Performance optimization: Debounce resize events
        let resizeTimeout;
        window.addEventListener('resize', function() {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(function() {
                // Adjust layout if needed
                console.log('Window resized, layout adjusted');
            }, 250);
        });

        // Initialize everything
        console.log('Cardano page initialized successfully');
    </script>
</body>
</html>