<?php
// Enable error reporting (for development only)
// Remove or disable in production
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in; if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$usd_balance    = isset($userDetails['balance']) ? (float)$userDetails['balance'] : 0.00;
$btc            = isset($userDetails['btc']) ? (float)$userDetails['btc'] : 0.000000;
$eth            = isset($userDetails['eth']) ? (float)$userDetails['eth'] : 0.000000;
$usdt           = isset($userDetails['usdt']) ? (float)$userDetails['usdt'] : 0.000000;
$xrp            = isset($userDetails['xrp']) ? (float)$userDetails['xrp'] : 0.000000;
$doge           = isset($userDetails['doge']) ? (float)$userDetails['doge'] : 0.000000;
$ada            = isset($userDetails['ada']) ? (float)$userDetails['ada'] : 0.000000;
$sol            = isset($userDetails['sol']) ? (float)$userDetails['sol'] : 0.000000;
$extra          = $userDetails['extra'];
$update_time    = $userDetails['update_time'];

// Define cache file and cache duration (2 hours)
$cache_file = "btc_price_cache.json";
$cache_duration = 7200; // 2 hours

// Function to fetch BTC price from CoinGecko API using cURL
function fetch_btc_price() {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=bitcoin&vs_currencies=usd";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Ignore SSL verification issues
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Timeout after 10 seconds
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Ensure response is valid and HTTP status code is 200 (OK)
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return false;
}

// Use cache if valid
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_duration) {
    $data = json_decode(file_get_contents($cache_file), true);
} else {
    // Fetch fresh BTC price
    $data = fetch_btc_price();

    // Store new price in cache if valid
    if ($data && isset($data['bitcoin']['usd'])) {
        file_put_contents($cache_file, json_encode($data));
    }
}

// Ensure BTC price is available
$btc_price = $data['bitcoin']['usd'] ?? null;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bitcoin (BTC) | primemarketspro.com</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <link rel="icon" href="https://primemarketspro.com/session/favicon.png" type="image/x-icon">
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <style>
        .btc-bg { background: linear-gradient(135deg, #F7931A 0%, #FFD700 100%); }
        .btc-text { color: #F7931A; }
        .btc-secondary { color: #FFD700; }
        .btn-btc {
            background: linear-gradient(135deg, #F7931A 0%, #FFD700 100%);
            color: white;
            transition: all 0.3s;
            border: none;
        }
        .btn-btc:hover {
            background: linear-gradient(135deg, #E6850E 0%, #F4D03F 100%);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(247, 147, 26, 0.4);
        }
        .tradingview-widget-container {
            width: 100%;
            min-height: 450px;
        }
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .progress-bar {
            height: 4px;
            background-color: #e2e8f0;
            border-radius: 2px;
            overflow: hidden;
        }
        .progress-value {
            height: 100%;
            background: linear-gradient(90deg, #F7931A 0%, #FFD700 100%);
            transition: width 0.5s ease;
        }
        .description-collapse {
            max-height: 100px;
            overflow: hidden;
            position: relative;
        }
        .description-collapse::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(to top, rgba(255,255,255,1), rgba(255,255,255,0));
        }
        .description-expanded {
            max-height: none;
        }
        .digital-gold-indicator {
            background: linear-gradient(135deg, #F7931A 0%, #FFD700 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
            margin-top: 4px;
        }
        .btc-card {
            background: linear-gradient(135deg, #FFF8E1 0%, #FFE0B2 100%);
            border: 2px solid #F7931A;
        }
        .price-bounce {
            animation: bounce 2s infinite;
        }
        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-10px); }
            60% { transform: translateY(-5px); }
        }
        .feature-tag {
            background: linear-gradient(135deg, #F7931A 0%, #FFD700 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .secure-tag {
            background: linear-gradient(135deg, #28A745 0%, #20C997 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .store-tag {
            background: linear-gradient(135deg, #6F42C1 0%, #E83E8C 100%);
            color: white;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: 600;
            display: inline-block;
            margin: 2px;
        }
        .performance-text {
            font-family: 'Courier New', monospace;
            color: #F7931A;
            font-weight: bold;
        }
        .glow-effect {
            box-shadow: 0 0 20px rgba(247, 147, 26, 0.3);
        }
        .hash-counter {
            font-family: 'Courier New', monospace;
            font-size: 18px;
            color: #FFD700;
            font-weight: bold;
            text-shadow: 0 0 10px rgba(255, 215, 0, 0.5);
        }
        .dominance-text {
            background: linear-gradient(135deg, #F7931A 0%, #FFD700 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            font-weight: bold;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen p-4">
    <div class="max-w-4xl mx-auto bg-white p-4 md:p-6 rounded-2xl shadow-sm">
        <!-- Header -->
        <header class="flex items-center justify-between mb-8">
            <button onclick="history.back()" class="btc-text p-2 rounded-full hover:bg-gray-100 transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                </svg>
            </button>
            <h1 class="text-xl font-semibold text-gray-800 flex items-center">
                <img src="https://primemarketspro.com/cryptos/bitcoin.png" class="w-8 h-8 mr-2" alt="Bitcoin">
                Bitcoin (BTC)
            </h1>
            <div class="p-2 rounded-full hover:bg-gray-100 transition-colors">
                <div id="connection-status" class="w-3 h-3 rounded-full bg-green-500"></div>
            </div>
        </header>

        <!-- Price Section -->
        <section class="text-center mb-8 fade-in">
            <div class="price-text text-4xl font-bold mb-2 flex items-center justify-center">
                <span id="btc-price" class="text-gray-800">$0.00</span>
                <span id="btc-arrow" class="ml-2"></span>
            </div>
            <div id="btc-change" class="text-lg font-medium text-gray-600">0.00% (24h)</div>
            <div class="digital-gold-indicator">
                🟡 Digital Gold
            </div>
            <div class="text-sm font-medium text-green-500 mt-2">
                1 USD ≈ <span id="btc-equivalent" class="text-gray-700">0.00000000 BTC</span>
            </div>
            <div class="flex justify-center mt-2 space-x-1">
                <span class="feature-tag">₿ P2P</span>
                <span class="secure-tag">🔒 Secure</span>
                <span class="store-tag">💎 Store of Value</span>
                <span class="feature-tag">Limited Supply</span>
            </div>
        </section>

        <!-- TradingView Widget Section -->
        <section class="mb-6" style="width: 100%;">
            <div class="tradingview-widget-container" style="width: 100%; height: 420px;">
                <div class="tradingview-widget-container__widget" style="width: 100%; height: 100%;"></div>
                <script type="text/javascript" src="https://s3.tradingview.com/external-embedding/embed-widget-mini-symbol-overview.js" async>
                {
                    "symbol": "BINANCE:BTCUSDT",
                    "width": "100%",
                    "height": "250",
                    "locale": "en",
                    "dateRange": "12M",
                    "colorTheme": "light",
                    "isTransparent": true,
                    "autosize": false,
                    "largeChartUrl": ""
                }
                </script>
            </div>
        </section>

        <!-- Wallet Card -->
        <section class="wallet-card btc-card rounded-xl p-6 mb-8 fade-in glow-effect">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm">Your BTC Balance</p>
                    <h2 class="text-2xl font-bold text-gray-800 mt-1">
                        <?php echo number_format($btc, 8); ?> BTC
                    </h2>
                    <p class="text-gray-600 mt-1">
                        ≈ $<span id="btc-balance-usd"><?php echo number_format($btc * 50000.00, 2); // Initial estimate ?></span> USD
                    </p>
                    <p class="performance-text text-xs mt-1">Digital store of value</p>
                </div>
                <img src="https://primemarketspro.com/cryptos/bitcoin.png" class="w-12 h-12" alt="Bitcoin">
            </div>
            <div class="mt-4 flex space-x-2">
                <a href="bitcoins.php" class="btn-btc py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-up mr-2"></i>Send
                </a>
                <a href="bitcoinr.php" class="btn-btc py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-down mr-2"></i>Receive
                </a>
            </div>
        </section>

        <!-- Network Stats -->
        <section class="bg-gradient-to-r from-orange-50 to-yellow-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Network Statistics</h3>
            <div class="grid grid-cols-2 gap-4">
                <div class="text-center">
                    <div class="hash-counter" id="hash-rate">~550 EH/s</div>
                    <div class="text-xs text-gray-500">Hash Rate</div>
                </div>
                <div class="text-center">
                    <div class="hash-counter">~10 min</div>
                    <div class="text-xs text-gray-500">Block Time</div>
                </div>
                <div class="text-center">
                    <div class="hash-counter" id="difficulty">~62.5T</div>
                    <div class="text-xs text-gray-500">Difficulty</div>
                </div>
                <div class="text-center">
                    <div class="hash-counter" id="next-halving">~2028</div>
                    <div class="text-xs text-gray-500">Next Halving</div>
                </div>
            </div>
        </section>

        <!-- Market Data -->
        <section class="bg-gray-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Market Overview</h3>
            <div class="space-y-4 text-sm">
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap Rank</span>
                        <span id="market-cap-rank">#1</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap</span>
                        <span id="market-cap">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>24h Volume</span>
                        <span id="volume">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Circulating Supply</span>
                        <span id="supply">0 BTC</span>
                    </div>
                    <div class="progress-bar mt-1">
                        <div id="supply-progress" class="progress-value" style="width: 0%"></div>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        <span id="circulating-supply">0</span> / <span id="max-supply">21M</span> BTC
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Dominance</span>
                        <span id="dominance" class="dominance-text">~55%</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Bitcoin's share of total crypto market
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Fear & Greed Index</span>
                        <span id="fear-greed" class="text-blue-600 font-medium">Neutral</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Market sentiment indicator
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Mining Reward</span>
                        <span id="mining-reward" class="text-orange-600 font-medium">3.125 BTC</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Current block reward
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Blocks Until Halving</span>
                        <span id="blocks-to-halving" class="text-purple-600 font-medium">~125,000</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Estimated blocks remaining
                    </div>
                </div>
            </div>
        </section>

        <!-- Security & Decentralization Info -->
        <section class="bg-gradient-to-r from-orange-50 to-yellow-50 border-2 border-orange-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-orange-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-orange-600 text-lg">🔒</span>
                </div>
                <div>
                    <h4 class="font-semibold text-orange-800 mb-1">Ultimate Security & Decentralization</h4>
                    <p class="text-sm text-orange-700">
                        Bitcoin operates on the most secure and decentralized network in the world, powered by Proof-of-Work consensus and protected by massive computational power.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-orange-600">✓ Most secure blockchain network</div>
                        <div class="text-xs text-orange-600">✓ Highest hash rate (~550 EH/s)</div>
                        <div class="text-xs text-orange-600">✓ 15+ years of uptime</div>
                        <div class="text-xs text-orange-600">✓ Immutable transaction history</div>
                        <div class="text-xs text-orange-600">✓ Global network of nodes</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- Store of Value Info -->
        <section class="bg-purple-50 border border-purple-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-purple-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <span class="text-purple-600 text-lg">💎</span>
                </div>
                <div>
                    <h4 class="font-semibold text-purple-800 mb-1">Digital Store of Value</h4>
                    <p class="text-sm text-purple-700">
                        Bitcoin is widely recognized as "digital gold" - a scarce, durable, and portable store of value that serves as a hedge against inflation and economic uncertainty.
                    </p>
                    <div class="mt-2 space-y-1">
                        <div class="text-xs text-purple-600">✓ Limited supply of 21 million BTC</div>
                        <div class="text-xs text-purple-600">✓ Institutional adoption growing</div>
                        <div class="text-xs text-purple-600">✓ Legal tender in some countries</div>
                        <div class="text-xs text-purple-600">✓ Inflation hedge properties</div>
                        <div class="text-xs text-purple-600">✓ Global accessibility 24/7</div>
                    </div>
                </div>
            </div>
        </section>

        <!-- About Section -->
        <section class="mb-8 fade-in">
            <div class="flex justify-between items-center mb-4">
                <h3 class="font-semibold text-gray-800">About Bitcoin</h3>
                <button id="toggle-description" class="text-orange-500 text-sm font-medium">Show More</button>
            </div>
            <div id="description-content" class="description-collapse text-gray-600 text-sm leading-relaxed">
                Loading Bitcoin description...
            </div>
        </section>

        <!-- Official Links -->
        <section class="fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Resources</h3>
            <div class="grid grid-cols-2 gap-3">
                <a href="https://bitcoin.org" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Bitcoin.org</div>
                </a>
                <a href="https://github.com/bitcoin/bitcoin" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Source Code</div>
                </a>
                <a href="blockchain_details.php" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Block Explorer</div>
                </a>
                <a href="https://bitcoin.org/en/developer-documentation" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Documentation</div>
                </a>
                <a href="https://bitcoinmagazine.com" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Bitcoin Magazine</div>
                </a>
                <a href="https://mempool.space" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-orange-500 font-medium">Mempool Stats</div>
                </a>
            </div>
        </section>
    </div>

    <script>
        // DOM Elements
        const btcPriceEl = document.getElementById('btc-price');
        const btcArrowEl = document.getElementById('btc-arrow');
        const btcChangeEl = document.getElementById('btc-change');
        const btcEquivalentEl = document.getElementById('btc-equivalent');
        const btcBalanceUsdEl = document.getElementById('btc-balance-usd');
        const marketCapEl = document.getElementById('market-cap');
        const volumeEl = document.getElementById('volume');
        const supplyEl = document.getElementById('supply');
        const marketCapRankEl = document.getElementById('market-cap-rank');
        const dominanceEl = document.getElementById('dominance');
        const fearGreedEl = document.getElementById('fear-greed');
        const miningRewardEl = document.getElementById('mining-reward');
        const blocksToHalvingEl = document.getElementById('blocks-to-halving');
        const descriptionContentEl = document.getElementById('description-content');
        const toggleDescriptionBtn = document.getElementById('toggle-description');
        const connectionStatusEl = document.getElementById('connection-status');
        const supplyProgressEl = document.getElementById('supply-progress');
        const circulatingSupplyEl = document.getElementById('circulating-supply');
        const maxSupplyEl = document.getElementById('max-supply');
        const hashRateEl = document.getElementById('hash-rate');
        const difficultyEl = document.getElementById('difficulty');
        const nextHalvingEl = document.getElementById('next-halving');

        // User data from PHP
        const userBTC = <?php echo json_encode($btc); ?>;
        let previousPrice = 0;
        let coinData = {};
        let descriptionExpanded = false;

        // Toggle description
        toggleDescriptionBtn.addEventListener('click', () => {
            descriptionExpanded = !descriptionExpanded;
            descriptionContentEl.classList.toggle('description-collapse', !descriptionExpanded);
            descriptionContentEl.classList.toggle('description-expanded', descriptionExpanded);
            toggleDescriptionBtn.textContent = descriptionExpanded ? 'Show Less' : 'Show More';
        });

        // Format numbers with commas
        function formatNumber(num, decimals = 0) {
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(num);
        }

        // Format currency
        function formatCurrency(amount) {
            return new Intl.NumberFormat('en-US', {
                style: 'currency',
                currency: 'USD',
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(amount);
        }

        // Update UI with price data
        function updatePriceUI(price, change) {
            // Update price
            btcPriceEl.textContent = formatCurrency(price);
            
            // Update 24h change
            const changeFixed = change.toFixed(2);
            btcChangeEl.textContent = `${changeFixed}% (24h)`;
            btcChangeEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update arrow
            btcArrowEl.textContent = change >= 0 ? '↑' : '↓';
            btcArrowEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update equivalent
            const btcEquivalent = 1 / price;
            btcEquivalentEl.textContent = btcEquivalent.toFixed(8) + ' BTC';
            
            // Update user balance
            const btcBalanceUSD = userBTC * price;
            btcBalanceUsdEl.textContent = formatNumber(btcBalanceUSD, 2);
            
            // Price change animation with bounce effect
            if (previousPrice && previousPrice !== price) {
                btcPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                void btcPriceEl.offsetWidth; // Trigger reflow
                btcPriceEl.classList.add(price > previousPrice ? 'text-green-500' : 'text-red-500');
                btcPriceEl.classList.add('price-bounce');
                
                setTimeout(() => {
                    btcPriceEl.classList.remove('text-green-500', 'text-red-500', 'price-bounce');
                    btcPriceEl.classList.add('text-gray-800');
                }, 3000);
            }
            
            previousPrice = price;
        }

        // Update market data
        function updateMarketData(data) {
            marketCapEl.textContent = formatCurrency(data.market_data.market_cap.usd);
            volumeEl.textContent = formatCurrency(data.market_data.total_volume.usd);
            supplyEl.textContent = formatNumber(data.market_data.circulating_supply / 1000000, 2) + 'M BTC';
            marketCapRankEl.textContent = '#' + data.market_cap_rank;
            
            // Update supply progress (Bitcoin has max supply of 21M)
            const circulating = data.market_data.circulating_supply;
            const maxSupply = 21000000;
            const supplyPercentage = (circulating / maxSupply) * 100;
            supplyProgressEl.style.width = supplyPercentage + '%';
            circulatingSupplyEl.textContent = formatNumber(circulating / 1000000, 2) + 'M';
            maxSupplyEl.textContent = '21M';
            
            // Update dominance (approximate)
            dominanceEl.textContent = '~55%';
        }

        // Update network stats
        function updateNetworkStats() {
            // These values would typically come from a blockchain API
            hashRateEl.textContent = '~550 EH/s';
            difficultyEl.textContent = '~62.5T';
            nextHalvingEl.textContent = '~2028';
        }

        // Fetch Bitcoin data from CoinGecko API
        async function fetchBitcoinData() {
            try {
                connectionStatusEl.style.backgroundColor = '#F59E0B'; // Orange - loading
                
                const response = await axios.get('https://api.coingecko.com/api/v3/coins/bitcoin');
                const data = response.data;
                coinData = data;
                
                // Update price and change
                const price = data.market_data.current_price.usd;
                const change = data.market_data.price_change_percentage_24h;
                updatePriceUI(price, change);
                
                // Update market data
                updateMarketData(data);
                
                // Update description
                descriptionContentEl.textContent = data.description.en || 'Bitcoin is the first decentralized digital currency, created in 2009 by an unknown person or group using the name Satoshi Nakamoto.';
                
                // Update network stats
                updateNetworkStats();
                
                connectionStatusEl.style.backgroundColor = '#10B981'; // Green - connected
                
            } catch (error) {
                console.error('Error fetching Bitcoin data:', error);
                connectionStatusEl.style.backgroundColor = '#EF4444'; // Red - error
                
                // Fallback to PHP cached price if available
                <?php if ($btc_price): ?>
                const fallbackPrice = <?php echo $btc_price; ?>;
                updatePriceUI(fallbackPrice, 0);
                <?php endif; ?>
                
                // Show error message
                descriptionContentEl.textContent = 'Unable to load latest Bitcoin data. Please check your internet connection.';
            }
        }

        // Update fear and greed index (placeholder function)
        function updateFearGreedIndex() {
            // This would typically come from a dedicated API
            const sentiments = ['Extreme Fear', 'Fear', 'Neutral', 'Greed', 'Extreme Greed'];
            const randomIndex = Math.floor(Math.random() * sentiments.length);
            fearGreedEl.textContent = sentiments[randomIndex];
            
            // Color coding
            const colors = ['#EF4444', '#F59E0B', '#6B7280', '#10B981', '#059669'];
            fearGreedEl.style.color = colors[randomIndex];
        }

        // Initialize the page
        function init() {
            // Fetch initial data
            fetchBitcoinData();
            updateFearGreedIndex();
            
            // Set up periodic updates (every 60 seconds)
            setInterval(fetchBitcoinData, 60000);
            setInterval(updateFearGreedIndex, 300000); // Update every 5 minutes
        }

        // Start the application when DOM is loaded
        document.addEventListener('DOMContentLoaded', init);

        // Handle visibility change to pause/resume updates
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                // Page is hidden, could pause updates
                connectionStatusEl.style.backgroundColor = '#6B7280'; // Gray - paused
            } else {
                // Page is visible, resume updates
                fetchBitcoinData();
            }
        });
    </script>
</body>
</html>