<?php
// Enable full error reporting (for development only, remove or disable in production)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in, if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);

if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id     = $userDetails['google_id'];
$name          = $userDetails['name'];
$email         = $userDetails['email'];
$created_at    = $userDetails['created_at'];
$picture       = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin           = $userDetails['pin'];
$status        = $userDetails['status'];
$country       = $userDetails['country'];
$phone         = $userDetails['phone'];
$balance       = $userDetails['balance'];
$btc           = $userDetails['btc'];
$eth           = $userDetails['eth'];
$usdt          = $userDetails['usdt'];
$xrp           = $userDetails['xrp'];
$message         = $userDetails['message'];
$update_time   = $userDetails['update_time'];
?>
<?php
// Enable error reporting for debugging (remove in production)

if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['email']);
    }
}

if (!function_exists('getUserDetails')) {
    function getUserDetails($conn) {
        $email = $_SESSION['email'] ?? null;
        if (!$email) return null;

        $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $result = $stmt->get_result();
        $userDetails = $result->fetch_assoc();
        $stmt->close();

        return $userDetails;
    }
}

if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

$userEmail = $userDetails['email'];

// Helper functions
function getStatusColor($status) {
    switch (strtolower($status)) {
        case 'completed':
        case 'confirmed':
            return 'bg-green-100 text-green-800';
        case 'pending':
        case 'unconfirmed':
            return 'bg-yellow-100 text-yellow-800'; // Changed from red to yellow
        case 'confirming':
            return 'bg-blue-100 text-blue-800';
        case 'declined':
        case 'rejected':
        case 'failed':
            return 'bg-red-100 text-red-800'; // Added declined status as red
        default:
            return 'bg-gray-100 text-gray-800';
    }
}

function truncateHash($hash) {
    return substr($hash, 0, 6) . '...' . substr($hash, -4);
}

function errorMessage($message) {
    return '<div class="text-red-500">' . htmlspecialchars($message) . '</div>';
}

function getNetworkIcon($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'bitcoin':
        case 'btc':
            return 'https://primemarketspro.com/cryptos/bitcoin.png';
        case 'ethereum':
        case 'eth':
            return 'https://primemarketspro.com/cryptos/eth.png';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'https://primemarketspro.com/cryptos/usdt.png';
        case 'xrp':
            return 'https://primemarketspro.com/cryptos/xrp.png';
        case 'dogecoin':
        case 'doge':
            return 'https://primemarketspro.com/cryptos/doge.png';
        case 'cardano':
        case 'ada':
            return 'https://primemarketspro.com/cryptos/ada.png';
        case 'bnb':
        case 'binance':
            return 'https://primemarketspro.com/cryptos/bnb.png';
        case 'pepe':
            return 'https://primemarketspro.com/cryptos/pepe.png';
        case 'sol':
            return 'https://primemarketspro.com/cryptos/sol.png';
        default:
            return 'https://cryptologos.cc/logos/measurable-data-token-mdt-logo.png';
    }
}

function getNetworkDisplayName($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'btc':
            return 'Bitcoin';
        case 'eth':
            return 'Ethereum';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'USDT';
        case 'xrp':
            return 'XRP';
        case 'doge':
        case 'dogecoin':
            return 'Dogecoin';
        case 'ada':
        case 'cardano':
            return 'Cardano';
        case 'bnb':
        case 'binance':
            return 'BNB';
        case 'pepe':
            return 'PEPE';
        case 'sol':
            return 'Solana';
        default:
            return ucfirst($network);
    }
}

function getCoinGeckoId($network) {
    $network = strtolower(trim($network));
    switch ($network) {
        case 'btc':
        case 'bitcoin':
            return 'bitcoin';
        case 'eth':
        case 'ethereum':
            return 'ethereum';
        case 'usdt':
        case 'erc20':
        case 'trc20':
            return 'tether';
        case 'xrp':
            return 'ripple';
        case 'doge':
        case 'dogecoin':
            return 'dogecoin';
        case 'ada':
        case 'cardano':
            return 'cardano';
        case 'bnb':
        case 'binance':
            return 'binancecoin';
        case 'pepe':
            return 'pepe';
        case 'sol':
            return 'solana';
        default:
            return null;
    }
}

// Helper function to format amounts with commas and decimals
function formatAmount($amount, $isCrypto = false) {
    if ($amount == 0) return '0';
    
    if ($isCrypto) {
        // Format crypto amounts with up to 8 decimals, remove trailing zeros
        $formatted = rtrim(number_format($amount, 8, '.', ','), '0');
        return rtrim($formatted, '.');
    }
    
    // Format USD amounts with 2 decimals and commas
    return number_format($amount, 2, '.', ',');
}

// AJAX endpoint for getting crypto prices
if (isset($_GET['get_prices'])) {
    header('Content-Type: application/json');
    
    $networks = ['bitcoin', 'ethereum', 'tether', 'ripple', 'dogecoin', 'cardano', 'binancecoin', 'pepe', 'solana'];
    $ids = implode(',', $networks);
    
    $url = "https://api.coingecko.com/api/v3/simple/price?ids={$ids}&vs_currencies=usd";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response !== false) {
        echo $response;
    } else {
        echo json_encode(['error' => 'Failed to fetch prices']);
    }
    exit();
}

// --- AJAX Endpoint ---
if (isset($_GET['get_transactions'])) {
    header('Content-Type: text/html');

    // Validate and assign GET parameters
    $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ["options" => ["default" => 1, "min_range" => 1]]);
    $perPage = filter_input(INPUT_GET, 'perPage', FILTER_VALIDATE_INT, ["options" => ["default" => 6, "min_range" => 1, "max_range" => 100]]);
    $offset = ($page - 1) * $perPage;

    $filterType = (isset($_GET['type']) && $_GET['type'] === 'received') ? 'received' : 'sent';
    $start = (isset($_GET['start']) && !empty($_GET['start'])) ? $_GET['start'] : null;
    $end   = (isset($_GET['end']) && !empty($_GET['end'])) ? $_GET['end'] : null;

    // Base query and parameters
    if ($filterType === 'sent') {
        $baseQuery = "SELECT id, email, total_amount AS amount, created_at, status, network, transhash 
                      FROM crypto_withdrawal WHERE email = ?";
    } else {
        $baseQuery = "SELECT id, email, amount, created_at, status, network, transhash 
                      FROM payments WHERE email = ?";
    }

    $conditions = '';
    $params = [$userEmail];
    $paramTypes = 's';

    if ($start) {
        $conditions .= " AND created_at >= ?";
        $params[] = $start . ' 00:00:00';
        $paramTypes .= 's';
    }
    if ($end) {
        $conditions .= " AND created_at <= ?";
        $params[] = $end . ' 23:59:59';
        $paramTypes .= 's';
    }

    if (!empty($conditions)) {
        $baseQuery .= $conditions;
    }

    $orderQuery = $baseQuery . " ORDER BY created_at DESC";

    // Count query
    $countQuery = "SELECT COUNT(*) FROM ($baseQuery) AS sub";

    try {
        $stmt = $conn->prepare($countQuery);
        $stmt->bind_param($paramTypes, ...$params);
        $stmt->execute();
        $stmt->bind_result($total);
        $stmt->fetch();
        $stmt->close();
    } catch (mysqli_sql_exception $e) {
        echo errorMessage('Count query failed: ' . $e->getMessage());
        exit();
    }

    // Data query
    $dataQuery = $orderQuery . " LIMIT ? OFFSET ?";
    $paramsForDataQuery = $params;
    $paramTypesForDataQuery = $paramTypes . 'ii';
    $paramsForDataQuery[] = $perPage;
    $paramsForDataQuery[] = $offset;

    try {
        $stmt = $conn->prepare($dataQuery);
        $stmt->bind_param($paramTypesForDataQuery, ...$paramsForDataQuery);
        $stmt->execute();
        $result = $stmt->get_result();
        $transactions = $result->fetch_all(MYSQLI_ASSOC);
        $stmt->close();
    } catch (mysqli_sql_exception $e) {
        echo errorMessage('Query failed: ' . $e->getMessage());
        exit();
    }

    // Output transactions
    ?>
    <div id="transactionsListContent" class="grid gap-6">
    <?php if (!empty($transactions)): ?>
        <?php foreach ($transactions as $tx): 
            $isSent = ($filterType === 'sent');
            $status = (floatval($tx['amount']) == 0.0) ? 'Confirming' : ($tx['status'] ?? 'Pending');
            $statusColor = getStatusColor($status);
            $amount = floatval($tx['amount']);
            $formattedAmount = formatAmount($amount, false); // USD formatting
            $date = new DateTime($tx['created_at']);
            $txHash = $tx['transhash'] ?? 'N/A';
            $network = htmlspecialchars($tx['network'] ?? 'Unknown');
            $networkDisplayName = getNetworkDisplayName($network);
            $coinGeckoId = getCoinGeckoId($network);
            $displayDate = htmlspecialchars($date->format('M d, Y H:i'));
            $transactionType = $isSent ? 'Sent' : 'Received';
            $networkIcon = getNetworkIcon($network);
        ?>
        <!-- Enhanced neumorphic card -->
        <a href="detailed.php?id=<?= urlencode($tx['id']) ?>&type=<?= urlencode($filterType) ?>" class="block">
            <div class="group flex flex-col md:flex-row items-center justify-between p-6 bg-gradient-to-br from-gray-50 to-white rounded-2xl hover:from-gray-100 hover:to-gray-50 transition-all duration-300 
                        shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)] 
                        hover:shadow-[12px_12px_24px_rgba(163,177,198,0.8),-12px_-12px_24px_rgba(255,255,255,0.9)]
                        border border-gray-100/50 backdrop-blur-sm">
                <div class="flex items-center space-x-5">
                    <!-- Enhanced neumorphic icon container -->
                    <div class="<?= $isSent ? 'bg-gradient-to-br from-red-50 to-red-100' : 'bg-gradient-to-br from-green-50 to-green-100' ?> 
                                p-4 rounded-2xl 
                                shadow-[inset_6px_6px_12px_rgba(163,177,198,0.3),inset_-6px_-6px_12px_rgba(255,255,255,0.9)]
                                border border-white/50">
                        <svg xmlns="http://www.w3.org/2000/svg" class="w-7 h-7 <?= $isSent ? 'text-red-500' : 'text-green-500' ?>" fill="currentColor" viewBox="0 0 24 24">
                            <?php if($isSent): ?>
                                <path d="M13 7.41V20a1 1 0 0 1-2 0V7.41l-5.3 5.3a1 1 0 0 1-1.4-1.42l7-7a1 1 0 0 1 1.4 0l7 7a1 1 0 0 1-1.4 1.42L13 7.4z"/>
                            <?php else: ?>
                                <path d="M11 16.59V4a1 1 0 0 1 2 0v12.59l5.3-5.3a1 1 0 0 1 1.4 1.42l-7 7a1 1 0 0 1-1.4 0l-7-7a1 1 0 0 1 1.4-1.42l5.3 5.3z"/>
                            <?php endif; ?>
                        </svg>
                    </div>
                    
                    <div>
                        <p class="font-semibold text-gray-800 text-lg">
                            <?= $transactionType ?>
                        </p>
                        <div class="flex items-center text-sm text-gray-600 mt-1">
                            <img src="<?= $networkIcon ?>" alt="<?= $networkDisplayName ?>" class="w-5 h-5 mr-2 rounded-full">
                            <?= $networkDisplayName ?>
                        </div>
                        <p class="text-sm text-gray-500 mt-1"><?= $displayDate ?></p>
                    </div>
                </div>
                
                <div class="text-right mt-4 md:mt-0">
                    <!-- Main display: USD amount from database -->
                    <p class="text-xl font-bold <?= $isSent ? 'text-red-600' : 'text-green-600' ?>">
                        <?php if(floatval($tx['amount']) == 0.0): ?>
                            <i class="fas fa-spinner fa-spin text-gray-400"></i>
                        <?php else: ?>
                            <?= ($isSent ? '-' : '+') . '$' . $formattedAmount ?>
                        <?php endif; ?>
                    </p>
                    
                    <!-- Crypto equivalent (calculated from USD) -->
                    <p class="text-sm <?= $isSent ? 'text-red-500' : 'text-green-500' ?> crypto-amount mt-1" 
                       data-usd-amount="<?= $amount ?>" 
                       data-network="<?= $coinGeckoId ?>" 
                       data-is-sent="<?= $isSent ? 'true' : 'false' ?>"
                       data-network-symbol="<?= strtoupper($network) ?>">
                        <?php if (floatval($tx['amount']) != 0.0): ?>
                            <!-- Crypto equivalent will be populated by JavaScript -->
                            <span class="text-gray-400">Converting...</span>
                        <?php endif; ?>
                    </p>
                    
                    <!-- Enhanced neumorphic status badge -->
                    <div class="inline-flex items-center mt-2 px-4 py-2 rounded-xl text-sm font-medium
                                shadow-[4px_4px_8px_rgba(163,177,198,0.4),-4px_-4px_8px_rgba(255,255,255,0.8)]
                                border border-white/50 <?= $statusColor ?>">
                        <span><?= htmlspecialchars(ucfirst($status)) ?></span>
                    </div>
                    
                    <p class="text-xs text-gray-500 mt-2 font-mono bg-gray-50 rounded-lg px-3 py-1 
                             shadow-[inset_2px_2px_4px_rgba(163,177,198,0.3),inset_-2px_-2px_4px_rgba(255,255,255,0.9)]">
                        <span title="<?= htmlspecialchars($txHash) ?>">Transhash:
                            <?= htmlspecialchars(truncateHash($txHash)) ?>
                        </span>
                    </p>
                </div>
            </div>
        </a>
        <?php endforeach; ?>
    <?php else: ?>
        <div class="text-center py-12 text-gray-500">
            <div class="bg-gradient-to-br from-gray-50 to-white rounded-2xl p-8 
                        shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.8)]
                        border border-gray-100/50">
                <i class="fas fa-coins text-4xl mb-4 text-gray-400"></i>
                <p class="text-lg">No transactions found</p>
            </div>
        </div>
    <?php endif; ?>
    </div>

    <?php
    // Enhanced neumorphic pagination with <1234> buttons
    $totalPages = ceil($total / $perPage);
    if ($totalPages > 1):
        // Calculate visible page range
        $visiblePages = 4; // Show up to 4 page numbers
        $startPage = max(1, min($page - floor($visiblePages/2), $totalPages - $visiblePages + 1));
        $endPage = min($startPage + $visiblePages - 1, $totalPages);
    ?>
    <div id="paginationContent" class="flex justify-center gap-3 mt-8">
        <?php if ($page > 1): ?>
        <button onclick="loadTransactions('<?= $filterType ?>', <?= $page-1 ?>)" 
                class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                       text-gray-700 transition-all duration-200
                       shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                       hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                       border border-gray-100/50">
            <i class="fas fa-chevron-left"></i>
        </button>
        <?php endif; ?>

        <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
            <?php if ($i == $page): ?>
                <button class="px-4 py-2 bg-gradient-to-br from-[#4669db] to-[#3b56c4] text-white rounded-xl 
                               shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                               border border-[#4669db]/20">
                    <?= $i ?>
                </button>
            <?php else: ?>
                <button onclick="loadTransactions('<?= $filterType ?>', <?= $i ?>)" 
                        class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                               text-gray-700 transition-all duration-200
                               shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                               hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                               border border-gray-100/50">
                    <?= $i ?>
                </button>
            <?php endif; ?>
        <?php endfor; ?>

        <?php if ($page < $totalPages): ?>
        <button onclick="loadTransactions('<?= $filterType ?>', <?= $page+1 ?>)" 
                class="px-4 py-2 bg-gradient-to-br from-white to-gray-50 rounded-xl hover:from-gray-50 hover:to-gray-100 
                       text-gray-700 transition-all duration-200
                       shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                       hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                       border border-gray-100/50">
            <i class="fas fa-chevron-right"></i>
        </button>
        <?php endif; ?>
    </div>
    <?php endif;
    exit();
}
?>

<?php include './includes/header.php'; ?>

<body class="bg-gradient-to-br from-gray-100 to-gray-200 min-h-screen">
  <?php include './includes/nav.php'; ?>
 
   <div style="padding: 4px; margin-top: 65px;">

       <!-- Main Content -->
       <div class="mb-24">
           <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-8">
               <div class="flex space-x-3 mb-4 md:mb-0">
                   <!-- Enhanced neumorphic filter buttons -->
                   <button id="filterSent" class="px-6 py-3 bg-gradient-to-br from-[#f0e8ff] to-[#e6d7ff] text-[#4669db] rounded-xl 
                                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                                  border border-[#4669db]/20 font-medium transition-all duration-200
                                                  hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]" 
                           onclick="setFilter('sent')">
                       Sent
                   </button>
                   <button id="filterReceived" class="px-6 py-3 bg-gradient-to-br from-white to-gray-50 border border-[#4669db]/30 text-[#4669db] rounded-xl 
                                                      shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                                      hover:from-[#f0e8ff] hover:to-[#e6d7ff] font-medium transition-all duration-200
                                                      hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]" 
                           onclick="setFilter('received')">
                       Received
                   </button>
               </div>
               <div class="flex space-x-3">
                   <!-- Enhanced neumorphic date inputs -->
                   <input type="date" id="startDate" value="<?= date('Y-m-d'); ?>" 
                          class="px-4 py-3 border border-[#4669db]/30 rounded-xl text-[#4669db] bg-gradient-to-br from-white to-gray-50
                                 shadow-[inset_4px_4px_8px_rgba(163,177,198,0.3),inset_-4px_-4px_8px_rgba(255,255,255,0.9)]
                                 focus:shadow-[inset_6px_6px_12px_rgba(163,177,198,0.4),inset_-6px_-6px_12px_rgba(255,255,255,1)]
                                 focus:outline-none focus:ring-2 focus:ring-[#4669db]/20 transition-all duration-200">
                   <input type="date" id="endDate" value="<?= date('Y-m-d'); ?>" 
                          class="px-4 py-3 border border-[#4669db]/30 rounded-xl text-[#4669db] bg-gradient-to-br from-white to-gray-50
                                 shadow-[inset_4px_4px_8px_rgba(163,177,198,0.3),inset_-4px_-4px_8px_rgba(255,255,255,0.9)]
                                 focus:shadow-[inset_6px_6px_12px_rgba(163,177,198,0.4),inset_-6px_-6px_12px_rgba(255,255,255,1)]
                                 focus:outline-none focus:ring-2 focus:ring-[#4669db]/20 transition-all duration-200">
                   <!-- Enhanced filter button -->
                   <button class="px-6 py-3 bg-gradient-to-br from-white to-gray-50 border border-[#4669db]/30 text-[#4669db] rounded-xl 
                                  shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                                  hover:from-[#f0e8ff] hover:to-[#e6d7ff] font-medium transition-all duration-200
                                  hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]" 
                           onclick="applyDateFilter()">
                       Filter
                   </button>
               </div>
           </div>

           <div id="transactionsContainer">
               <!-- Loading indicator -->
               <div class="text-center py-12">
                   <i class="fas fa-spinner fa-spin text-2xl text-blue-500"></i>
                   <p class="text-gray-600 mt-2">Loading transactions...</p>
               </div>
           </div>
       </div>
   </div>

<?php
require('db.php'); // Ensure database connection is set up

// Retrieve the user email from session
$user_email = getSessionEmail();

if (!$user_email) {
    echo '<div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded relative" role="alert">
            <strong class="font-bold">Not Logged In</strong>
            <span class="block sm:inline"> Please log in to access your dashboard.</span>
          </div>';
    exit;
}

// Fetch user details (msgstatus, message)
$query = "SELECT msgstatus, message FROM users WHERE email = :email";
$stmt = $conn->prepare($query);
$stmt->bindParam(':email', $user_email, PDO::PARAM_STR);
$stmt->execute();
$user = $stmt->fetch(PDO::FETCH_ASSOC);

$show_modal = false;
$popup_message = '';

if ($user) {
    $msgstatus = strtolower(trim($user['msgstatus'] ?? ''));
    $message = trim($user['message'] ?? '');

    // Show popup only if msgstatus is "yes"
    if ($msgstatus === 'yes') {
        $show_modal = true;
        $popup_message = $message; // Fetching 'message' value from the database
    }
}
?>
<!-- Enhanced neumorphic popup overlay -->
<div id="overlay" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.4); backdrop-filter: blur(4px); z-index: 999;"></div>

<!-- Enhanced neumorphic popup modal -->
<div id="popup" style="display: none; position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); 
                      background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); 
                      border-radius: 20px; padding: 30px; width: 90%; max-width: 500px; z-index: 1000;
                      box-shadow: 20px 20px 40px rgba(163,177,198,0.6), -20px -20px 40px rgba(255,255,255,0.8);
                      border: 1px solid rgba(255,255,255,0.5);">
    <div class="text-center">
        <div class="mb-6">
            <div class="inline-flex items-center justify-center w-16 h-16 bg-gradient-to-br from-blue-50 to-blue-100 rounded-full
                        shadow-[inset_6px_6px_12px_rgba(163,177,198,0.3),inset_-6px_-6px_12px_rgba(255,255,255,0.9)]
                        border border-white/50 mb-4">
                <i class="fas fa-info-circle text-2xl text-blue-500"></i>
            </div>
            <h2 class="text-xl font-bold text-gray-800 mb-4">Important Notice</h2>
            <p class="text-gray-600 text-sm leading-relaxed" id="popup-message">
                <?= htmlspecialchars($popup_message) ?>
            </p>
        </div>
        <button onclick="closePopup()" 
                class="px-8 py-3 bg-gradient-to-br from-blue-500 to-blue-600 text-white rounded-xl font-medium
                       shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)]
                       hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]
                       hover:from-blue-600 hover:to-blue-700 transition-all duration-200">
            Got it
        </button>
    </div>
</div>

<script>
// Global variables
let currentFilter = 'sent';
let cryptoPrices = {};

// Show popup on page load if needed
<?php if ($show_modal): ?>
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('overlay').style.display = 'block';
    document.getElementById('popup').style.display = 'block';
});
<?php endif; ?>

// Close popup function
function closePopup() {
    document.getElementById('overlay').style.display = 'none';
    document.getElementById('popup').style.display = 'none';
}

// Fetch crypto prices
async function fetchCryptoPrices() {
    try {
        const response = await fetch('?get_prices=1');
        const data = await response.json();
        
        if (!data.error) {
            cryptoPrices = data;
            updateCryptoAmounts();
        }
    } catch (error) {
        console.error('Error fetching crypto prices:', error);
    }
}

// Update crypto amounts in the UI
function updateCryptoAmounts() {
    const cryptoElements = document.querySelectorAll('.crypto-amount');
    
    cryptoElements.forEach(element => {
        const usdAmount = parseFloat(element.dataset.usdAmount);
        const network = element.dataset.network;
        const isSent = element.dataset.isSent === 'true';
        const networkSymbol = element.dataset.networkSymbol;
        
        if (usdAmount > 0 && cryptoPrices[network]) {
            const cryptoPrice = cryptoPrices[network].usd;
            const cryptoAmount = usdAmount / cryptoPrice;
            const formattedCrypto = cryptoAmount.toFixed(8).replace(/\.?0+$/, '');
            
            element.innerHTML = `${isSent ? '-' : '+'}${formattedCrypto} ${networkSymbol}`;
        }
    });
}

// Set filter and update UI
function setFilter(type) {
    currentFilter = type;
    
    // Update button states
    const sentBtn = document.getElementById('filterSent');
    const receivedBtn = document.getElementById('filterReceived');
    
    if (type === 'sent') {
        sentBtn.className = 'px-6 py-3 bg-gradient-to-br from-[#f0e8ff] to-[#e6d7ff] text-[#4669db] rounded-xl shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)] border border-[#4669db]/20 font-medium transition-all duration-200 hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]';
        receivedBtn.className = 'px-6 py-3 bg-gradient-to-br from-white to-gray-50 border border-[#4669db]/30 text-[#4669db] rounded-xl shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)] hover:from-[#f0e8ff] hover:to-[#e6d7ff] font-medium transition-all duration-200 hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]';
    } else {
        sentBtn.className = 'px-6 py-3 bg-gradient-to-br from-white to-gray-50 border border-[#4669db]/30 text-[#4669db] rounded-xl shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)] hover:from-[#f0e8ff] hover:to-[#e6d7ff] font-medium transition-all duration-200 hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]';
        receivedBtn.className = 'px-6 py-3 bg-gradient-to-br from-[#f0e8ff] to-[#e6d7ff] text-[#4669db] rounded-xl shadow-[6px_6px_12px_rgba(163,177,198,0.5),-6px_-6px_12px_rgba(255,255,255,0.8)] border border-[#4669db]/20 font-medium transition-all duration-200 hover:shadow-[8px_8px_16px_rgba(163,177,198,0.6),-8px_-8px_16px_rgba(255,255,255,0.9)]';
    }
    
    loadTransactions(type, 1);
}

// Load transactions via AJAX
function loadTransactions(type, page = 1) {
    const startDate = document.getElementById('startDate').value;
    const endDate = document.getElementById('endDate').value;
    
    const url = `?get_transactions=1&type=${type}&page=${page}&start=${startDate}&end=${endDate}`;
    
    fetch(url)
        .then(response => response.text())
        .then(html => {
            document.getElementById('transactionsContainer').innerHTML = html;
            // Update crypto amounts after loading transactions
            setTimeout(() => {
                updateCryptoAmounts();
            }, 100);
        })
        .catch(error => {
            console.error('Error loading transactions:', error);
            document.getElementById('transactionsContainer').innerHTML = 
                '<div class="text-center py-12 text-red-500">Error loading transactions</div>';
        });
}

// Apply date filter
function applyDateFilter() {
    loadTransactions(currentFilter, 1);
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    // Load initial transactions
    loadTransactions('sent', 1);
    
    // Fetch crypto prices
    fetchCryptoPrices();
    
    // Refresh prices every 60 seconds
    setInterval(fetchCryptoPrices, 60000);
});

// Add event listeners for date inputs
document.addEventListener('DOMContentLoaded', function() {
    const startDate = document.getElementById('startDate');
    const endDate = document.getElementById('endDate');
    
    startDate.addEventListener('change', applyDateFilter);
    endDate.addEventListener('change', applyDateFilter);
});
</script>

<?php include './includes/footer.php'; ?>
</body>
</html>