<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Redirect with an error message.
 *
 * @param string $message The error message to set in the session.
 */
function redirectWithError($message) {
    $_SESSION['error'] = $message;
    header("Location: dashboard.php");
    exit();
}

// Ensure the request method is POST.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: dashboard.php");
    exit();
}

// Validate required fields.
$required = ['walletAddress', 'network', 'email', 'name'];
foreach ($required as $field) {
    if (empty($_POST[$field])) {
        redirectWithError("Missing required field: $field");
    }
}

// Sanitize input values.
$wallet_address = htmlspecialchars($_POST['walletAddress'], ENT_QUOTES, 'UTF-8');
$network        = htmlspecialchars($_POST['network'], ENT_QUOTES, 'UTF-8');
$amount         = isset($_POST['amount']) ? $_POST['amount'] : 'confirming';
$email          = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
$name           = htmlspecialchars($_POST['name'], ENT_QUOTES, 'UTF-8');
$ip_address     = $_SERVER['REMOTE_ADDR'];

// Validate email format.
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    redirectWithError("Invalid email format");
}

// Process $amount: if it’s not 'confirming', cast to float.
if ($amount !== 'confirming') {
    $amount = (float) $amount;
}

// Debug: Log file upload details for 'prove'.
if (isset($_FILES['prove'])) {
    error_log("File details: " . print_r($_FILES['prove'], true));
    error_log("File upload error code: " . $_FILES['prove']['error']);
}

/**
 * Handles file upload.
 *
 * @param string $fileInputName Name of the file input.
 * @param string $uploadDir Absolute path of the upload directory.
 * @param array  $allowedExtensions Allowed file extensions.
 * @param int    $maxSize Maximum allowed file size in bytes.
 * @return string Returns the new file name (relative path) if successful, or an empty string.
 */
function handleFileUpload($fileInputName, $uploadDir, $allowedExtensions, $maxSize) {
    if (isset($_FILES[$fileInputName]) && $_FILES[$fileInputName]['error'] === UPLOAD_ERR_OK) {
        // Check file size.
        if ($_FILES[$fileInputName]['size'] > $maxSize) {
            redirectWithError("File is too large. Maximum size is " . ($maxSize / (1024 * 1024)) . "MB.");
        }
        $fileTmpPath = $_FILES[$fileInputName]['tmp_name'];
        $fileName    = $_FILES[$fileInputName]['name'];
        
        // Get file extension and validate.
        $fileNameComponents = explode(".", $fileName);
        $fileExtension = strtolower(end($fileNameComponents));
        if (!in_array($fileExtension, $allowedExtensions)) {
            redirectWithError("File type not allowed. Allowed types: " . implode(', ', $allowedExtensions));
        }
        
        // Generate a secure, unique file name.
        $newFileName = bin2hex(random_bytes(16)) . '.' . $fileExtension;
        
        // Create the upload directory if it doesn't exist.
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0755, true)) {
                error_log("Failed to create upload directory: " . $uploadDir);
                redirectWithError("Failed to create upload directory.");
            }
            chmod($uploadDir, 0755);
        }
        
        // Verify that the directory is writable.
        if (!is_writable($uploadDir)) {
            error_log("Upload directory is not writable: " . $uploadDir);
            redirectWithError("Upload directory is not writable.");
        }
        
        $destPath = $uploadDir . $newFileName;
        if (move_uploaded_file($fileTmpPath, $destPath)) {
            chmod($destPath, 0644); // Set secure file permissions.
            error_log("File successfully uploaded to: " . $destPath);
            return $newFileName; // Return the relative file name for storage.
        } else {
            error_log("Error moving uploaded file. Tmp Path: $fileTmpPath, Destination: $destPath");
            error_log("Current directory permissions: " . substr(sprintf('%o', fileperms($uploadDir)), -4));
            error_log("PHP user: " . exec('whoami'));
            redirectWithError("There was an error moving the uploaded file.");
        }
    } else if (isset($_FILES[$fileInputName]) && $_FILES[$fileInputName]['error'] !== UPLOAD_ERR_NO_FILE) {
        // Convert PHP file upload error code to human-readable message.
        $uploadErrors = [
            UPLOAD_ERR_INI_SIZE   => "The uploaded file exceeds the upload_max_filesize directive in php.ini",
            UPLOAD_ERR_FORM_SIZE  => "The uploaded file exceeds the MAX_FILE_SIZE directive in the HTML form",
            UPLOAD_ERR_PARTIAL    => "The uploaded file was only partially uploaded",
            UPLOAD_ERR_NO_TMP_DIR => "Missing a temporary folder",
            UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk",
            UPLOAD_ERR_EXTENSION  => "A PHP extension stopped the file upload"
        ];
        $errorCode = $_FILES[$fileInputName]['error'];
        $errorMessage = isset($uploadErrors[$errorCode]) ? $uploadErrors[$errorCode] : "Unknown upload error";
        error_log("File upload error: " . $errorMessage);
        redirectWithError("File upload failed: " . $errorMessage);
    }
    return "";
}

// Process file upload for 'prove'.
$uploadDir = __DIR__ . '/uploads/';
$allowedExtensions = ['jpg', 'jpeg', 'png', 'pdf', 'doc', 'docx', 'txt'];
$maxFileSize = 10 * 1024 * 1024; // 10MB limit.
$provePath = handleFileUpload('prove', $uploadDir, $allowedExtensions, $maxFileSize);

// --- Database Insertion Section ---

// It is best practice to store credentials in a separate configuration file or environment variables.
$dbHost = "localhost";
$dbUser = "wptlgmnb_userp";
$dbPass = "Qliox4ITygj6Vh4V";
$dbName = "wptlgmnb_dbp";

// Create a new MySQLi connection.
$conn = new mysqli($dbHost, $dbUser, $dbPass, $dbName);
if ($conn->connect_error) {
    redirectWithError("Database connection failed. Please try again later.");
}

// Generate a unique transaction hash.
$transhash = hash('sha256', $email . microtime(true) . random_int(1000, 9999));
$status = "pending";

// Prepare SQL statement for insertion.
$sql = "INSERT INTO payments (
    wallet_address, 
    network, 
    amount, 
    email, 
    name, 
    ip_address,
    status,
    prove,
    created_at, 
    transhash
) VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?)";
$stmt = $conn->prepare($sql);
if (!$stmt) {
    redirectWithError("Prepare statement failed: " . $conn->error);
}

// Bind parameters based on whether $amount is 'confirming' or a numeric value.
if ($amount === 'confirming') {
    $stmt->bind_param("sssssssss", 
        $wallet_address,
        $network,
        $amount,
        $email,
        $name,
        $ip_address,
        $status,
        $provePath,
        $transhash
    );
} else {
    $stmt->bind_param("ssdssssss", 
        $wallet_address,
        $network,
        $amount,
        $email,
        $name,
        $ip_address,
        $status,
        $provePath,
        $transhash
    );
}

// Execute the statement and check for errors.
if (!$stmt->execute()) {
    error_log("Database insertion error: " . $stmt->error);
    redirectWithError("Database insertion failed: " . $stmt->error);
}
$stmt->close();

// --- Email Sending Section using PHPMailer ---

require '../vendor/autoload.php';
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

$mail = new PHPMailer(true);
try {
    $mail->isSMTP();
    $mail->Host       = 'mail.primemarketspro.com';
    $mail->SMTPAuth   = true;
    $mail->Username   = 'noreply@primemarketspro.com';
    $mail->Password   = 'Trexn@504';
    $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
    $mail->Port       = 587;

    $mail->setFrom('noreply@primemarketspro.com', 'primemarketspro.com Wallet');
    $mail->addAddress($email, $name);
    
    // Attach the uploaded file if available.
    $attachmentPath = __DIR__ . '/uploads/' . $provePath;
    if (!empty($provePath) && file_exists($attachmentPath)) {
        $mail->addAttachment($attachmentPath);
    }
    
    $mail->isHTML(true);
    $amountDisplay = $amount === 'confirming' ? 'confirming' : '$' . number_format($amount, 2) . ' USD';
    $mail->Subject = 'Transaction Confirmation - primemarketspro.com Wallet';
    $mail->Body = "<html><body>
        <p>Dear " . htmlspecialchars($name, ENT_QUOTES, 'UTF-8') . ",</p>
        <p>Your transaction was successful. Details:</p>
        <ul>
            <li><strong>Transaction ID:</strong> $transhash</li>
            <li><strong>Status:</strong> $status</li>
            <li><strong>Network:</strong> " . htmlspecialchars($network, ENT_QUOTES, 'UTF-8') . "</li>
            <li><strong>Wallet Address:</strong> " . htmlspecialchars($wallet_address, ENT_QUOTES, 'UTF-8') . "</li>
            <li><strong>Amount:</strong> $amountDisplay</li>
        </ul>
        <p>Thank you for using primemarketspro.com Wallet.</p>
        <p>&copy; " . date('Y') . " primemarketspro.com Wallet. All rights reserved.</p>
        </body></html>";
    $mail->AltBody = "Transaction Confirmation - primemarketspro.com Wallet\n\nDear $name,\n\nYour transaction was successful. Details:\n\n- Transaction ID: $transhash\n- Status: $status\n- Network: $network\n- Wallet Address: $wallet_address\n- Amount: $amountDisplay\n\nThank you for using primemarketspro.com Wallet.";
    $mail->send();
    error_log("Email sent successfully to: $email");
} catch (Exception $e) {
    error_log("Email sending error: " . $mail->ErrorInfo);
    redirectWithError("Email could not be sent: " . $mail->ErrorInfo);
}

$conn->close();
$_SESSION['success'] = "Transaction processed successfully!";
header("Location: dashboard.php");
exit();
?>
