<?php
// Enable full error reporting (for development only)
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in, if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);

if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$name          = htmlspecialchars($userDetails['name']);
$email         = htmlspecialchars($userDetails['email']);
$picture       = htmlspecialchars($userDetails['picture']);
$pin           = htmlspecialchars($userDetails['pin']);
$country       = htmlspecialchars($userDetails['country']);
$phone         = htmlspecialchars($userDetails['phone']);
$twofa_enabled = isset($userDetails['2fa_enabled']) ? $userDetails['2fa_enabled'] : 0;
$twofa_secret  = isset($userDetails['2fa_secret']) ? $userDetails['2fa_secret'] : '';

// Include Google Authenticator library
require_once '../vendor/autoload.php'; // Assuming you have composer installed
use Google\Authenticator\GoogleAuthenticator;
$ga = new GoogleAuthenticator();

// Generate 2FA secret if not already set
if (empty($twofa_secret)) {
    $twofa_secret = $ga->createSecret();
    // Save the secret to database for future reference
    $stmt = $conn->prepare("UPDATE users SET 2fa_secret = ? WHERE id = ?");
    $stmt->bind_param("si", $twofa_secret, $_SESSION['user_id']);
    $stmt->execute();
    $stmt->close();
}

// Generate QR code URL
$qrCodeUrl = $ga->getQRCodeGoogleUrl(
    'primemarketspro.com Wallet:' . $email,
    $twofa_secret,
    'primemarketspro.com Wallet'
);

// Initialize message and error variables
$successMessage = "";
$errors = [];
$passwordSuccessMessage = "";
$passwordErrors = [];
$twoFASuccessMessage = "";
$twoFAErrors = [];
$deletionSuccessMessage = "";
$deletionErrors = [];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Determine which form was submitted
    if (isset($_POST['update_profile'])) {
        // Profile update form
        $newName    = trim($_POST['name']);
        $newPin     = trim($_POST['pin']);
        $newCountry = trim($_POST['country']);
        $newPhone   = trim($_POST['phone']);

        // Validate inputs
        if (empty($newName)) {
            $errors[] = "Name is required.";
        }
        if (empty($newPin)) {
            $errors[] = "PIN is required.";
        } elseif (!is_numeric($newPin)) {
            $errors[] = "PIN must be a number.";
        }
        if (empty($newCountry)) {
            $errors[] = "Country is required.";
        }
        // Phone is optional, no validation needed

        // If no errors, update the database
        if (empty($errors)) {
            $userId = $_SESSION['user_id']; // Assuming user_id is stored in the session
            $sql = "UPDATE users SET name = ?, pin = ?, country = ?, phone = ? WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("ssssi", $newName, $newPin, $newCountry, $newPhone, $userId);
                if ($stmt->execute()) {
                    $successMessage = "Profile updated successfully!";
                    // Refresh user details after update
                    $userDetails = getUserDetails($conn);
                    $name    = htmlspecialchars($userDetails['name']);
                    $pin     = htmlspecialchars($userDetails['pin']);
                    $country = htmlspecialchars($userDetails['country']);
                    $phone   = htmlspecialchars($userDetails['phone']);
                } else {
                    $errors[] = "Failed to update profile. Please try again.";
                }
                $stmt->close();
            } else {
                $errors[] = "Database error. Please try again.";
            }
        }
    } elseif (isset($_POST['change_password'])) {
        // Password change form
        $currentPassword = trim($_POST['current_password']);
        $newPassword = trim($_POST['new_password']);
        $confirmPassword = trim($_POST['confirm_password']);

        // Validate inputs
        if (empty($currentPassword)) {
            $passwordErrors[] = "Current password is required.";
        }
        if (empty($newPassword)) {
            $passwordErrors[] = "New password is required.";
        } elseif (strlen($newPassword) < 8) {
            $passwordErrors[] = "New password must be at least 8 characters.";
        }
        if ($newPassword !== $confirmPassword) {
            $passwordErrors[] = "New passwords do not match.";
        }

        // If no validation errors, verify current password and update
        if (empty($passwordErrors)) {
            $userId = $_SESSION['user_id'];
            
            // First, get the current hashed password
            $sql = "SELECT password FROM users WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("i", $userId);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($row = $result->fetch_assoc()) {
                    $hashedPassword = $row['password'];
                    
                    // Verify the current password
                    if (password_verify($currentPassword, $hashedPassword)) {
                        // Password matches, now update with the new one
                        $newHashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                        
                        $updateSql = "UPDATE users SET password = ? WHERE id = ?";
                        $updateStmt = $conn->prepare($updateSql);
                        if ($updateStmt) {
                            $updateStmt->bind_param("si", $newHashedPassword, $userId);
                            if ($updateStmt->execute()) {
                                $passwordSuccessMessage = "Password changed successfully!";
                            } else {
                                $passwordErrors[] = "Failed to update password. Please try again.";
                            }
                            $updateStmt->close();
                        } else {
                            $passwordErrors[] = "Database error. Please try again.";
                        }
                    } else {
                        $passwordErrors[] = "Current password is incorrect.";
                    }
                } else {
                    $passwordErrors[] = "User not found.";
                }
                $stmt->close();
            } else {
                $passwordErrors[] = "Database error. Please try again.";
            }
        }
    } elseif (isset($_POST['enable_2fa'])) {
        // Enable 2FA form submission
        $verificationCode = trim($_POST['verification_code']);
        
        // Verify the code submitted by user
        if ($ga->checkCode($twofa_secret, $verificationCode)) {
            // Code is valid, enable 2FA for the user
            $userId = $_SESSION['user_id'];
            $sql = "UPDATE users SET 2fa_enabled = 1 WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("i", $userId);
                if ($stmt->execute()) {
                    $twofa_enabled = 1;
                    $twoFASuccessMessage = "Two-factor authentication has been enabled successfully!";
                } else {
                    $twoFAErrors[] = "Failed to enable 2FA. Please try again.";
                }
                $stmt->close();
            }
        } else {
            $twoFAErrors[] = "Invalid verification code. Please try again.";
        }
    } elseif (isset($_POST['disable_2fa'])) {
        // Disable 2FA form submission
        $verificationCode = trim($_POST['verification_code_disable']);
        
        // Verify the code submitted by user
        if ($ga->checkCode($twofa_secret, $verificationCode)) {
            // Code is valid, disable 2FA for the user
            $userId = $_SESSION['user_id'];
            $sql = "UPDATE users SET 2fa_enabled = 0 WHERE id = ?";
            $stmt = $conn->prepare($sql);
            if ($stmt) {
                $stmt->bind_param("i", $userId);
                if ($stmt->execute()) {
                    $twofa_enabled = 0;
                    $twoFASuccessMessage = "Two-factor authentication has been disabled.";
                } else {
                    $twoFAErrors[] = "Failed to disable 2FA. Please try again.";
                }
                $stmt->close();
            }
        } else {
            $twoFAErrors[] = "Invalid verification code. Please try again.";
        }
    } elseif (isset($_POST['request_account_deletion'])) {
        // Account deletion request
        $reason = trim($_POST['deletion_reason']);
        $confirmEmail = trim($_POST['confirm_email']);
        
        // Validate inputs
        if (empty($reason)) {
            $deletionErrors[] = "Please provide a reason for account deletion.";
        }
        
        if ($confirmEmail !== $email) {
            $deletionErrors[] = "Email address does not match your account email.";
        }
        
        if (empty($deletionErrors)) {
            // Create a deletion request in the database
            $userId = $_SESSION['user_id'];
            $status = 'pending'; // pending, approved, rejected
            $requestDate = date('Y-m-d H:i:s');
            
            // First check if there's already a pending request
            $checkSql = "SELECT id FROM deletion_requests WHERE user_id = ? AND status = 'pending'";
            $checkStmt = $conn->prepare($checkSql);
            $checkStmt->bind_param("i", $userId);
            $checkStmt->execute();
            $result = $checkStmt->get_result();
            
            if ($result->num_rows > 0) {
                $deletionErrors[] = "You already have a pending deletion request. Please wait for admin approval.";
            } else {
                // Insert new deletion request
                $sql = "INSERT INTO deletion_requests (user_id, reason, request_date, status) VALUES (?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                
                if ($stmt) {
                    $stmt->bind_param("isss", $userId, $reason, $requestDate, $status);
                    
                    if ($stmt->execute()) {
                        $deletionSuccessMessage = "Account deletion request submitted successfully. Your request will be reviewed by our team.";
                    } else {
                        $deletionErrors[] = "Failed to submit deletion request. Please try again.";
                    }
                    $stmt->close();
                } else {
                    $deletionErrors[] = "Database error. Please try again.";
                }
            }
            $checkStmt->close();
        }
    }
}

// Check if user has a pending deletion request
$hasPendingDeletionRequest = false;
$deletionRequestStatus = '';

$userId = $_SESSION['user_id'];
$checkSql = "SELECT status FROM deletion_requests WHERE user_id = ? ORDER BY request_date DESC LIMIT 1";
$checkStmt = $conn->prepare($checkSql);

if ($checkStmt) {
    $checkStmt->bind_param("i", $userId);
    $checkStmt->execute();
    $result = $checkStmt->get_result();
    
    if ($row = $result->fetch_assoc()) {
        $deletionRequestStatus = $row['status'];
        if ($deletionRequestStatus === 'pending') {
            $hasPendingDeletionRequest = true;
        }
    }
    $checkStmt->close();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>primemarketspro.com Wallet - User Profile</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
    /* Base Styles */
    body {
      font-family: 'Inter', sans-serif;
      background-color: #F2F5FA;
      color: #1F2937;
      overflow-x: hidden;
      margin: 0;
      padding: 0;
    }
    /* Preloader Styles */
    .preloader {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background-color: #F2F5FA;
      display: flex;
      justify-content: center;
      align-items: center;
      z-index: 9999;
    }
    .loader {
      border: 6px solid #E5E7EB;
      border-top: 6px solid #8b63f1;
      border-radius: 50%;
      width: 40px;
      height: 40px;
      animation: spin 1s linear infinite;
    }
    @keyframes spin {
      to { transform: rotate(360deg); }
    }
    /* Scrollbar Styles (optional) */
    ::-webkit-scrollbar {
      width: 8px;
    }
    ::-webkit-scrollbar-thumb {
      background-color: #CBD5E1;
      border-radius: 9999px;
    }
    /* Card Styles */
    .card {
      background-color: #FFFFFF;
      border-radius: 16px;
      box-shadow: 0 10px 15px rgba(0, 0, 0, 0.05);
    }
    /* Button Styles */
    .btn-gradient {
      background: linear-gradient(90deg, #4F46E5 0%, #7C3AED 100%);
      color: #FFFFFF;
      box-shadow: 0 4px 6px rgba(124, 58, 237, 0.4);
    }
    .btn-icon {
      background-color: #E0E7FF;
      color: #4F46E5;
    }
    /* Responsive Adjustments */
    @media (max-width: 640px) {
      .nav-item-text {
        display: none;
      }
    }
    /* Modal Styles */
    .modal {
      display: none;
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background-color: rgba(0, 0, 0, 0.5);
      z-index: 100;
      overflow: auto;
    }
    .modal-content {
      background-color: #FFFFFF;
      margin: 10% auto;
      padding: 20px;
      border-radius: 10px;
      max-width: 500px;
      box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
      position: relative;
    }
    .close {
      position: absolute;
      right: 20px;
      top: 15px;
      font-size: 24px;
      font-weight: bold;
      cursor: pointer;
    }
  </style>
  
    <!-- Favicon -->
    <link rel="icon" href="https://primemarketspro.com/session/favicon.png" type="image/x-icon">
</head>
<body class="bg-gray-50">
    <!-- Preloader -->
    <div id="preloader" class="fixed inset-0 bg-white flex items-center justify-center z-50">
        <div id="spinner" class="animate-spin rounded-full h-16 w-16 border-t-4 border-blue-600"></div>
    </div>
    
    <!-- Header -->
    <header id="main-header" class="bg-white shadow fixed w-full top-0 z-50 transition-colors duration-300">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex">
                    <!-- Logo -->
                    <div class="flex-shrink-0 flex items-center">
                        <img class="h-8 w-auto" src="https://primemarketspro.com/session/favicon.png" alt="primemarketspro.com Logo">
                        <span class="ml-2 font-bold text-xl">Primemarketspro</span>
                    </div>
                    <!-- Desktop Navigation -->
                    <nav class="hidden sm:flex sm:items-center sm:ml-6 sm:space-x-8">
                        <a href="./dashboard.php" class="text-gray-500 hover:text-gray-700 hover:border-gray-300 inline-flex items-center px-1 pt-1 border-b-2 border-transparent text-sm font-medium">
                            Home
                        </a>
                        <a href="./history.php" class="text-gray-500 hover:text-gray-700 hover:border-gray-300 inline-flex items-center px-1 pt-1 border-b-2 border-transparent text-sm font-medium">
                            History
                        </a>
                        <!-- Receive Dropdown Desktop -->
                        <div class="relative">
                            <button id="desktop-receive-btn" class="text-gray-500 hover:text-gray-700 hover:border-gray-300 inline-flex items-center px-1 pt-1 border-b-2 border-transparent text-sm font-medium focus:outline-none">
                                Receive
                                <svg class="ml-1 h-4 w-4" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 011.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"/>
                                </svg>
                            </button>
                            <ul id="desktop-receive" class="absolute left-0 mt-2 w-40 bg-white shadow-lg rounded-md py-1 z-10 hidden">
                                <li>
                                    <a href="./bitcoinr.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Bitcoin
                                    </a>
                                </li>
                                <li>
                                    <a href="./ethr.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Ethereum (ETH)
                                    </a>
                                </li>
                                <li>
                                    <a href="./usdtr.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Tether (USDT)
                                    </a>
                                </li>
                                <li>
                                    <a href="./xrpr.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Ripple (XRP)
                                    </a>
                                </li>
                            </ul>
                        </div>
                        <!-- Send Dropdown Desktop -->
                        <div class="relative">
                            <button id="desktop-send-btn" class="text-gray-500 hover:text-gray-700 hover:border-gray-300 inline-flex items-center px-1 pt-1 border-b-2 border-transparent text-sm font-medium focus:outline-none">
                                Send
                                <svg class="ml-1 h-4 w-4" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 011.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"/>
                                </svg>
                            </button>
                            <ul id="desktop-send" class="absolute left-0 mt-2 w-40 bg-white shadow-lg rounded-md py-1 z-10 hidden">
                                <li>
                                    <a href="./bitcoins.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Bitcoin
                                    </a>
                                </li>
                                <li>
                                    <a href="./eths.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Ethereum (ETH)
                                    </a>
                                </li>
                                <li>
                                    <a href="./usdtd.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Tether (USDT)
                                    </a>
                                </li>
                                <li>
                                    <a href="./xrps.php" target="_blank" class="block px-4 py-2 text-gray-700 hover:bg-gray-100">
                                        Ripple (XRP)
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </nav>
                </div>
                <!-- Mobile Menu Button -->
                <div class="flex items-center sm:hidden">
                    <button id="mobile-menu-button" type="button" class="inline-flex items-center justify-center p-2 rounded-md text-gray-400 hover:text-gray-500 hover:bg-gray-100 focus:outline-none focus:bg-gray-100 focus:text-gray-500">
                        <span class="sr-only">Open main menu</span>
                        <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 6h16M4 12h16M4 18h16"/>
                        </svg>
                    </button>
                </div>
            </div>
        </div>
        <!-- Mobile Menu -->
        <div id="mobile-menu" class="sm:hidden hidden">
            <div class="pt-2 pb-3 space-y-1">
                <a href="./dashboard.php" class="block pl-3 pr-4 py-2 text-base font-medium text-gray-600 hover:bg-gray-50 hover:text-gray-800 hover:border-gray-300 border-l-4 border-transparent">
                    Home
                </a>
                <a href="./history.php" class="block pl-3 pr-4 py-2 text-base font-medium text-gray-600 hover:bg-gray-50 hover:text-gray-800 hover:border-gray-300 border-l-4 border-transparent">
                    History
                </a>
                <!-- Mobile Dropdown for Receive -->
                <div class="border-t border-gray-200">
                    <button id="mobile-receive-btn" type="button" class="w-full flex justify-between items-center pl-3 pr-4 py-2 text-base font-medium text-gray-600 hover:bg-gray-50 hover:text-gray-800 focus:outline-none">
                        Receive
                        <svg class="h-5 w-5 transform transition-transform duration-200" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 011.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"/>
                        </svg>
                    </button>
                    <ul id="mobile-receive" class="pl-4 hidden">
                        <li>
                            <a href="./bitcoinr.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Bitcoin
                            </a>
                        </li>
                        <li>
                            <a href="./ethr.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Ethereum (ETH)
                            </a>
                        </li>
                        <li>
                            <a href="./usdtr.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Tether (USDT)
                            </a>
                        </li>
                        <li>
                            <a href="./xrpr.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Ripple (XRP)
                            </a>
                        </li>
                    </ul>
                </div>
                <!-- Mobile Dropdown for Send -->
                <div class="border-t border-gray-200">
                    <button id="mobile-send-btn" type="button" class="w-full flex justify-between items-center pl-3 pr-4 py-2 text-base font-medium text-gray-600 hover:bg-gray-50 hover:text-gray-800 focus:outline-none">
                        Send
                        <svg class="h-5 w-5 transform transition-transform duration-200" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 011.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z" clip-rule="evenodd"/>
                        </svg>
                    </button>
                    <ul id="mobile-send" class="pl-4 hidden">
                        <li>
                            <a href="./bitcoins.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Bitcoin
                            </a>
                        </li>
                        <li>
                            <a href="./eths.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Ethereum (ETH)
                            </a>
                        </li>
                        <li>
                            <a href="./usdtd.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Tether (USDT)
                            </a>
                        </li>
                        <li>
                            <a href="./xrps.php" target="_blank" class="block pl-3 pr-4 py-2 text-gray-600 hover:bg-gray-100">
                                Ripple (XRP)
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </header>

    <div style="padding: 7px; margin-top: 76px;">
        <!-- Main Content -->
        <main class="container mx-auto px-4 py-8">
            <div class="flex items-center justify-between mb-6">
                <h1 class="text-2xl font-semibold text-gray-800">User Profile</h1>
                <div class="flex space-x-2">
                    <a href="dashboard.php" class="bg-blue-100 text-blue-600 px-3 py-1 rounded-md hover:bg-blue-200">
                        <i class="fas fa-chevron-left mr-1"></i> Back to Dashboard
                    </a>
                </div>
            </div>
            
            <!-- Tabs Navigation -->
            <div class="mb-6 border-b border-gray-200">
                <ul class="flex flex-wrap -mb-px text-sm font-medium text-center">
                    <li class="mr-2">
                        <a href="#profile-tab" class="tab-link inline-block p-4 border-b-2 border-blue-600 rounded-t-lg text-blue-600 active" data-tab="profile-tab">
                            <i class="fas fa-user mr-2"></i>Profile
                        </a>
                    </li>
                    <li class="mr-2">
                        <a href="#security-tab" class="tab-link inline-block p-4 border-b-2 border-transparent rounded-t-lg hover:text-gray-600 hover:border-gray-300" data-tab="security-tab">
                            <i class="fas fa-shield-alt mr-2"></i>Security
                        </a>
                    </li>
                    <li class="mr-2">
                        <a href="#account-tab" class="tab-link inline-block p-4 border-b-2 border-transparent rounded-t-lg hover:text-gray-600 hover:border-gray-300" data-tab="account-tab">
                            <i class="fas fa-cog mr-2"></i>Account
                        </a>
                    </li>
                </ul>
            </div>
            
            <!-- Tab Content -->
            <div class="tab-content">
                <!-- Profile Tab -->
                <div id="profile-tab" class="tab-pane active">
                    <div class="grid grid-cols-1 md:grid-cols-1 gap-6">
                        <!-- Profile Information Card -->
                        <div class="bg-white rounded-lg shadow-sm p-6">
                            <div class="text-center mb-6">
                                <img src="<?= $picture ?>" alt="User Picture" class="w-24 h-24 rounded-full mb-3 mx-auto">
                                <h2 class="text-xl font-semibold"><?= $name ?></h2>
                                <p class="text-gray-500"><?= $email ?></p>
                            </div>

                            <!-- Display success or error messages for profile update -->
                            <?php if (!empty($successMessage)): ?>
                                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                    <?= $successMessage; ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($errors)): ?>
                                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                                    <ul>
                                        <?php foreach ($errors as $error): ?>
                                            <li><?= $error; ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <!-- Profile Update Form -->
                            <form action="" method="POST">
                                <input type="hidden" name="update_profile" value="1">
                                <div class="mb-4">
                                    <label for="name" class="block text-gray-700">Name</label>
                                    <input type="text" name="name" id="name" value="<?= $name ?>" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <div class="mb-4">
                                    <label for="pin" class="block text-gray-700">PIN</label>
                                    <input type="text" name="pin" id="pin" value="<?= $pin ?>" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <div class="mb-4">
                                    <label for="country" class="block text-gray-700">Country</label>
                                    <input type="text" name="country" id="country" value="<?= $country ?>" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <div class="mb-4">
                                    <label for="phone" class="block text-gray-700">Phone</label>
                                    <input type="text" name="phone" id="phone" value="<?= $phone ?>" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <button type="submit" class="bg-blue-500 text-white px-4 py-2 rounded">Update Profile</button>
                            </form>
                        </div>
                    </div>
                </div>
                
                <!-- Security Tab -->
                <div id="security-tab" class="tab-pane hidden">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Password Change Card -->
                        <div class="bg-white rounded-lg shadow-sm p-6">
                            <h2 class="text-xl font-semibold mb-4">Change Password</h2>
                            <?php if (!empty($passwordSuccessMessage)): ?>
                                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                    <?= $passwordSuccessMessage; ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($passwordErrors)): ?>
                                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                                    <ul>
                                        <?php foreach ($passwordErrors as $error): ?>
                                            <li><?= $error; ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>
                            <form action="" method="POST">
                                <input type="hidden" name="change_password" value="1">
                                <div class="mb-4">
                                    <label for="current_password" class="block text-gray-700">Current Password</label>
                                    <input type="password" name="current_password" id="current_password" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <div class="mb-4">
                                    <label for="new_password" class="block text-gray-700">New Password</label>
                                    <input type="password" name="new_password" id="new_password" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <div class="mb-4">
                                    <label for="confirm_password" class="block text-gray-700">Confirm New Password</label>
                                    <input type="password" name="confirm_password" id="confirm_password" class="w-full border rounded px-3 py-2 mt-1">
                                </div>
                                <button type="submit" class="bg-blue-500 text-white px-4 py-2 rounded">Change Password</button>
                            </form>
                        </div>
                        <!-- 2FA Settings Card -->
                        <div class="bg-white rounded-lg shadow-sm p-6">
                            <h2 class="text-xl font-semibold mb-4">Two-Factor Authentication (2FA)</h2>
                            <?php if ($twofa_enabled): ?>
                                <p class="mb-4 text-green-600">2FA is currently enabled on your account.</p>
                                <?php if (!empty($twoFASuccessMessage)): ?>
                                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                        <?= $twoFASuccessMessage; ?>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($twoFAErrors)): ?>
                                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                                        <ul>
                                            <?php foreach ($twoFAErrors as $error): ?>
                                                <li><?= $error; ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                <form action="" method="POST">
                                    <input type="hidden" name="disable_2fa" value="1">
                                    <div class="mb-4">
                                        <label for="verification_code_disable" class="block text-gray-700">Enter 2FA Code to Disable</label>
                                        <input type="text" name="verification_code_disable" id="verification_code_disable" class="w-full border rounded px-3 py-2 mt-1">
                                    </div>
                                    <button type="submit" class="bg-red-500 text-white px-4 py-2 rounded">Disable 2FA</button>
                                </form>
                            <?php else: ?>
                                <p class="mb-4 text-red-600">2FA is currently disabled on your account.</p>
                                <?php if (!empty($twoFASuccessMessage)): ?>
                                    <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                        <?= $twoFASuccessMessage; ?>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($twoFAErrors)): ?>
                                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                                        <ul>
                                            <?php foreach ($twoFAErrors as $error): ?>
                                                <li><?= $error; ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    </div>
                                <?php endif; ?>
                                <div class="mb-4">
                                    <p>Scan this QR code with your authenticator app:</p>
                                    <img src="<?= $qrCodeUrl ?>" alt="2FA QR Code" class="mx-auto my-4">
                                    <p class="text-center">Or use this secret: <strong><?= $twofa_secret ?></strong></p>
                                </div>
                                <form action="" method="POST">
                                    <input type="hidden" name="enable_2fa" value="1">
                                    <div class="mb-4">
                                        <label for="verification_code" class="block text-gray-700">Enter Verification Code</label>
                                        <input type="text" name="verification_code" id="verification_code" class="w-full border rounded px-3 py-2 mt-1">
                                    </div>
                                    <button type="submit" class="bg-green-500 text-white px-4 py-2 rounded">Enable 2FA</button>
                                </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Account Tab -->
                <div id="account-tab" class="tab-pane hidden">
                    <div class="bg-white rounded-lg shadow-sm p-6">
                        <h2 class="text-xl font-semibold mb-4">Account Deletion</h2>
                        <?php if ($hasPendingDeletionRequest): ?>
                            <p class="mb-4 text-yellow-600">You have a pending account deletion request. Current status: <strong><?= htmlspecialchars($deletionRequestStatus) ?></strong>.</p>
                        <?php else: ?>
                            <?php if (!empty($deletionSuccessMessage)): ?>
                                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                                    <?= $deletionSuccessMessage; ?>
                                </div>
                            <?php endif; ?>
                            <?php if (!empty($deletionErrors)): ?>
                                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                                    <ul>
                                        <?php foreach ($deletionErrors as $error): ?>
                                            <li><?= $error; ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>
                            <form action="" method="POST">
                                <input type="hidden" name="request_account_deletion" value="1">
                                <div class="mb-4">
                                    <label for="deletion_reason" class="block text-gray-700">Reason for Account Deletion</label>
                                    <textarea name="deletion_reason" id="deletion_reason" rows="4" class="w-full border rounded px-3 py-2 mt-1"></textarea>
                                </div>
                                <div class="mb-4">
                                    <label for="confirm_email" class="block text-gray-700">Confirm your Email Address</label>
                                    <input type="email" name="confirm_email" id="confirm_email" class="w-full border rounded px-3 py-2 mt-1" value="<?= $email ?>">
                                </div>
                                <button type="submit" class="bg-red-600 text-white px-4 py-2 rounded">Request Account Deletion</button>
                            </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>
    
    <!-- JavaScript for interactivity -->
    <script>
        // Preloader removal
        window.addEventListener('load', function () {
            var preloader = document.getElementById('preloader');
            preloader.style.display = 'none';
        });

        // Tabs functionality
        const tabLinks = document.querySelectorAll('.tab-link');
        const tabPanes = document.querySelectorAll('.tab-pane');

        tabLinks.forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                // Remove active class from all links and hide all panes
                tabLinks.forEach(item => item.classList.remove('border-blue-600', 'text-blue-600', 'active'));
                tabPanes.forEach(pane => pane.classList.add('hidden'));

                // Add active class to clicked link and show associated pane
                this.classList.add('border-blue-600', 'text-blue-600', 'active');
                const tabId = this.getAttribute('data-tab');
                document.getElementById(tabId).classList.remove('hidden');
            });
        });

        // Mobile menu toggle
        document.getElementById('mobile-menu-button').addEventListener('click', function() {
            var mobileMenu = document.getElementById('mobile-menu');
            mobileMenu.classList.toggle('hidden');
        });

        // Mobile dropdowns toggle
        document.getElementById('mobile-receive-btn').addEventListener('click', function() {
            var dropdown = document.getElementById('mobile-receive');
            dropdown.classList.toggle('hidden');
        });
        document.getElementById('mobile-send-btn').addEventListener('click', function() {
            var dropdown = document.getElementById('mobile-send');
            dropdown.classList.toggle('hidden');
        });

        // Desktop dropdowns toggle
        document.getElementById('desktop-receive-btn').addEventListener('click', function() {
            var dropdown = document.getElementById('desktop-receive');
            dropdown.classList.toggle('hidden');
        });
        document.getElementById('desktop-send-btn').addEventListener('click', function() {
            var dropdown = document.getElementById('desktop-send');
            dropdown.classList.toggle('hidden');
        });
    </script>
</body>
</html>
