<?php
session_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Validate session and transaction data
if (
    !isset($_SESSION['transaction_data']) || 
    !isset($_SESSION['transaction_data']['network'], 
           $_SESSION['transaction_data']['total_amount'],
           $_SESSION['transaction_data']['wallet'],
           $_SESSION['transaction_data']['fee'])
) {
    header("Location: bitcoins.php");
    exit();
}

$transaction_data = $_SESSION['transaction_data'];

// Validate numeric values
if (!is_numeric($transaction_data['total_amount']) || $transaction_data['total_amount'] <= 0) {
    die("Invalid transaction amount");
}

// Fetch exchange rates with fallback
function fetchExchangeRates() {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=bitcoin,ethereum,tether,ripple,solana,cardano,binancecoin,pepe,dogecoin&vs_currencies=usd";
    $response = @file_get_contents($api_url);
    
    return $response ? json_decode($response, true) : [
        'bitcoin'      => ['usd' => 30000],
        'ethereum'     => ['usd' => 2000],
        'tether'       => ['usd' => 1],
        'ripple'       => ['usd' => 0.5],
        'solana'       => ['usd' => 40],
        'cardano'      => ['usd' => 1.2],
        'binancecoin'  => ['usd' => 300],
        'pepe'         => ['usd' => 0.00001],
        'dogecoin'     => ['usd' => 0.1]
    ];
}

$exchange_rates = fetchExchangeRates();
$network_type = strtolower($transaction_data['network']);

// Normalize network type for Binance Coin
if ($network_type === 'bnb') {
    $network_type = 'binancecoin';
}

$usd_amount = (float)$transaction_data['total_amount'];

// Crypto configuration including Solana, Cardano, and Binance Coin (BNB)
$crypto_config = [
    'bitcoin'      => ['symbol' => 'BTC', 'decimals' => 8],
    'ethereum'     => ['symbol' => 'ETH', 'decimals' => 6],
    'tether'       => ['symbol' => 'USDT', 'decimals' => 2],
    'ripple'       => ['symbol' => 'xrp', 'decimals' => 2],
    'solana'       => ['symbol' => 'SOL', 'decimals' => 2],
    'cardano'      => ['symbol' => 'ADA', 'decimals' => 2],
    'binancecoin'  => ['symbol' => 'BNB', 'decimals' => 2],
    'pepe'         => ['symbol' => 'PEPE', 'decimals' => 2],
    'dogecoin'     => ['symbol' => 'DOGE', 'decimals' => 2]
];

$config = $crypto_config[$network_type] ?? ['symbol' => 'USD', 'decimals' => 2];
$rate = $exchange_rates[$network_type]['usd'] ?? 1;
$crypto_amount = $rate > 0 ? $usd_amount / $rate : $usd_amount;

// Formatting
$formatted_amount = number_format($crypto_amount, $config['decimals']);
$crypto_symbol = $config['symbol'];

// Crypto icons (added ADA and BNB icons)
$crypto_icons = [
    'BTC'  => 'https://primemarketspro.com/cryptos/bitcoin.png',
    'ETH'  => 'https://primemarketspro.com/cryptos/eth.png',
    'USDT' => 'https://primemarketspro.com/cryptos/usdt.png',
    'xrp'  => 'https://primemarketspro.com/cryptos/xrp.png',
    'SOL'  => 'https://primemarketspro.com/cryptos/sol.png',
    'BNB'  => 'https://primemarketspro.com/cryptos/bnb.png',
    'PEPE' => 'https://primemarketspro.com/cryptos/pepe.png',
    'DOGE' => 'https://primemarketspro.com/cryptos/doge.png',
    'ADA'  => 'https://primemarketspro.com/cryptos/ada.png'
];
$crypto_icon = $crypto_icons[$crypto_symbol] ?? './favicon.png';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Confirmation</title>
    <link rel="icon" href="<?= $crypto_icon ?>" type="image/png">
    <!-- Favicon -->
    <link rel="icon" href="https://primemarketspro.com/session/faviconb.png" type="image/x-icon">
  
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <script>
        // Redirect after 25 seconds
        setTimeout(function() {
            window.location.href = "dashboard.php";
        }, 25000);
    </script>
</head>
<body class="bg-gray-100">
    <div class="flex flex-col items-center justify-center min-h-screen p-4">
        <div class="bg-white rounded-2xl shadow-xl w-full max-w-md p-8 animate__animated animate__fadeInUp">
            <div class="text-center space-y-4">
                <!-- Header Section -->
                <div class="flex items-center justify-center gap-2 mb-6">
                    <img src="<?= $crypto_icon ?>" alt="<?= $crypto_symbol ?> logo" class="w-8 h-8">
                    <span class="bg-green-100 text-green-800 px-3 py-1 rounded-full text-sm font-medium">
                        Payment Completed
                    </span>
                </div>

                <!-- Amount Display -->
                <h2 class="text-xl font-semibold text-gray-800">Successfully Sent</h2>
                <p class="text-3xl font-bold text-gray-900">
                    <?= $formatted_amount ?> 
                    <span class="text-gray-600"><?= $crypto_symbol ?></span>
                </p>

                <!-- Transaction Details -->
                <div class="text-sm text-gray-600 space-y-1">
                    <!-- If wallet address is too long, break it into multiple lines -->
                    <p>To: <?= nl2br(wordwrap(htmlspecialchars($transaction_data['wallet']), 30, "\n", true)) ?></p>
                    <p>Network Fee: <?= htmlspecialchars($transaction_data['fee']) ?> <?= $crypto_symbol ?></p>
                </div>

                <!-- Animated Checkmark -->
                <div class="flex items-center justify-center my-8 relative">
                    <div class="absolute w-32 h-32 rounded-full bg-green-200/50 animate-ping"></div>
                    <div class="w-20 h-20 rounded-full border-4 border-green-500 flex items-center justify-center bg-white relative">
                        <svg class="w-10 h-10 text-green-600 animate__animated animate__fadeIn" 
                             fill="none" 
                             stroke="currentColor" 
                             stroke-width="2" 
                             viewBox="0 0 24 24">
                            <path stroke-linecap="round" 
                                  stroke-linejoin="round" 
                                  d="M5 13l4 4L19 7"></path>
                        </svg>
                    </div>
                </div>

                <!-- Redirect Notice -->
                <p class="text-gray-500 text-sm">Redirecting in 25 seconds...</p>
            </div>
        </div>
    </div>
</body>
</html>
