<?php
// Enable error reporting for debugging (remove these in production)
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

/**
 * Map full country names to their ISO 3166-1 alpha-2 codes.
 * This mapping covers countries from A to Z.
 *
 * @param string $countryName The full country name.
 * @return string Two-letter country code in lowercase or an empty string if not found.
 */
function getCountryCode($countryName) {
    $codes = [
        'Afghanistan'                    => 'af',
        'Albania'                        => 'al',
        'Algeria'                        => 'dz',
        'Andorra'                        => 'ad',
        'Angola'                         => 'ao',
        'Antigua and Barbuda'            => 'ag',
        'Argentina'                      => 'ar',
        'Armenia'                        => 'am',
        'Australia'                      => 'au',
        'Austria'                        => 'at',
        'Azerbaijan'                     => 'az',
        'Bahamas'                        => 'bs',
        'Bahrain'                        => 'bh',
        'Bangladesh'                     => 'bd',
        'Barbados'                       => 'bb',
        'Belarus'                        => 'by',
        'Belgium'                        => 'be',
        'Belize'                         => 'bz',
        'Benin'                          => 'bj',
        'Bhutan'                         => 'bt',
        'Bolivia'                        => 'bo',
        'Bosnia and Herzegovina'         => 'ba',
        'Botswana'                       => 'bw',
        'Brazil'                         => 'br',
        'Brunei'                         => 'bn',
        'Bulgaria'                       => 'bg',
        'Burkina Faso'                   => 'bf',
        'Burundi'                        => 'bi',
        'Cabo Verde'                     => 'cv',
        'Cambodia'                       => 'kh',
        'Cameroon'                       => 'cm',
        'Canada'                         => 'ca',
        'Central African Republic'       => 'cf',
        'Chad'                           => 'td',
        'Chile'                          => 'cl',
        'China'                          => 'cn',
        'Colombia'                       => 'co',
        'Comoros'                        => 'km',
        'Congo (Congo-Brazzaville)'        => 'cg',
        'Costa Rica'                     => 'cr',
        'Croatia'                        => 'hr',
        'Cuba'                           => 'cu',
        'Cyprus'                         => 'cy',
        'Czechia (Czech Republic)'       => 'cz',
        'Democratic Republic of the Congo' => 'cd',
        'Denmark'                        => 'dk',
        'Djibouti'                       => 'dj',
        'Dominica'                       => 'dm',
        'Dominican Republic'             => 'do',
        'Ecuador'                        => 'ec',
        'Egypt'                          => 'eg',
        'El Salvador'                    => 'sv',
        'Equatorial Guinea'              => 'gq',
        'Eritrea'                        => 'er',
        'Estonia'                        => 'ee',
        'Eswatini'                       => 'sz',
        'Ethiopia'                       => 'et',
        'Fiji'                           => 'fj',
        'Finland'                        => 'fi',
        'France'                         => 'fr',
        'Gabon'                          => 'ga',
        'Gambia'                         => 'gm',
        'Georgia'                        => 'ge',
        'Germany'                        => 'de',
        'Ghana'                          => 'gh',
        'Greece'                         => 'gr',
        'Grenada'                        => 'gd',
        'Guatemala'                      => 'gt',
        'Guinea'                         => 'gn',
        'Guinea-Bissau'                  => 'gw',
        'Guyana'                         => 'gy',
        'Haiti'                          => 'ht',
        'Holy See'                       => 'va',
        'Honduras'                       => 'hn',
        'Hungary'                        => 'hu',
        'Iceland'                        => 'is',
        'India'                          => 'in',
        'Indonesia'                      => 'id',
        'Iran'                           => 'ir',
        'Iraq'                           => 'iq',
        'Ireland'                        => 'ie',
        'Israel'                         => 'il',
        'Italy'                          => 'it',
        'Jamaica'                        => 'jm',
        'Japan'                          => 'jp',
        'Jordan'                         => 'jo',
        'Kazakhstan'                     => 'kz',
        'Kenya'                          => 'ke',
        'Kiribati'                       => 'ki',
        'Kuwait'                         => 'kw',
        'Kyrgyzstan'                     => 'kg',
        'Laos'                           => 'la',
        'Latvia'                         => 'lv',
        'Lebanon'                        => 'lb',
        'Lesotho'                        => 'ls',
        'Liberia'                        => 'lr',
        'Libya'                          => 'ly',
        'Liechtenstein'                  => 'li',
        'Lithuania'                      => 'lt',
        'Luxembourg'                     => 'lu',
        'Madagascar'                     => 'mg',
        'Malawi'                         => 'mw',
        'Malaysia'                       => 'my',
        'Maldives'                       => 'mv',
        'Mali'                           => 'ml',
        'Malta'                          => 'mt',
        'Marshall Islands'               => 'mh',
        'Mauritania'                     => 'mr',
        'Mauritius'                      => 'mu',
        'Mexico'                         => 'mx',
        'Micronesia'                     => 'fm',
        'Moldova'                        => 'md',
        'Monaco'                         => 'mc',
        'Mongolia'                       => 'mn',
        'Montenegro'                     => 'me',
        'Morocco'                        => 'ma',
        'Mozambique'                     => 'mz',
        'Myanmar (formerly Burma)'       => 'mm',
        'Namibia'                        => 'na',
        'Nauru'                          => 'nr',
        'Nepal'                          => 'np',
        'Netherlands'                    => 'nl',
        'New Zealand'                    => 'nz',
        'Nicaragua'                      => 'ni',
        'Niger'                          => 'ne',
        'Nigeria'                        => 'ng',
        'North Korea'                    => 'kp',
        'North Macedonia'                => 'mk',
        'Norway'                         => 'no',
        'Oman'                           => 'om',
        'Pakistan'                       => 'pk',
        'Palau'                          => 'pw',
        'Palestine State'                => 'ps',
        'Panama'                         => 'pa',
        'Papua New Guinea'               => 'pg',
        'Paraguay'                       => 'py',
        'Peru'                           => 'pe',
        'Philippines'                    => 'ph',
        'Poland'                         => 'pl',
        'Portugal'                       => 'pt',
        'Qatar'                          => 'qa',
        'Romania'                        => 'ro',
        'Russia'                         => 'ru',
        'Rwanda'                         => 'rw',
        'Saint Kitts and Nevis'          => 'kn',
        'Saint Lucia'                    => 'lc',
        'Saint Vincent and the Grenadines'=> 'vc',
        'Samoa'                          => 'ws',
        'San Marino'                     => 'sm',
        'Sao Tome and Principe'          => 'st',
        'Saudi Arabia'                   => 'sa',
        'Senegal'                        => 'sn',
        'Serbia'                         => 'rs',
        'Seychelles'                     => 'sc',
        'Sierra Leone'                   => 'sl',
        'Singapore'                      => 'sg',
        'Slovakia'                       => 'sk',
        'Slovenia'                       => 'si',
        'Solomon Islands'                => 'sb',
        'Somalia'                        => 'so',
        'South Africa'                   => 'za',
        'South Korea'                    => 'kr',
        'South Sudan'                    => 'ss',
        'Spain'                          => 'es',
        'Sri Lanka'                      => 'lk',
        'Sudan'                          => 'sd',
        'Suriname'                       => 'sr',
        'Sweden'                         => 'se',
        'Switzerland'                    => 'ch',
        'Syria'                          => 'sy',
        'Taiwan'                         => 'tw',
        'Tajikistan'                     => 'tj',
        'Tanzania'                       => 'tz',
        'Thailand'                       => 'th',
        'Timor-Leste'                    => 'tl',
        'Togo'                           => 'tg',
        'Tonga'                          => 'to',
        'Trinidad and Tobago'            => 'tt',
        'Tunisia'                        => 'tn',
        'Turkey'                         => 'tr',
        'Turkmenistan'                   => 'tm',
        'Tuvalu'                         => 'tv',
        'Uganda'                         => 'ug',
        'Ukraine'                        => 'ua',
        'United Arab Emirates'           => 'ae',
        'United Kingdom'                 => 'gb',
        'United States'                  => 'us',
        'Uruguay'                        => 'uy',
        'Uzbekistan'                     => 'uz',
        'Vanuatu'                        => 'vu',
        'Venezuela'                      => 've',
        'Vietnam'                        => 'vn',
        'Yemen'                          => 'ye',
        'Zambia'                         => 'zm',
        'Zimbabwe'                       => 'zw'
    ];
    
    // If the country name is already 2 characters, assume it's already the code.
    if (strlen($countryName) == 2) return strtolower($countryName);
    return isset($codes[$countryName]) ? $codes[$countryName] : '';
}

require 'config.php';

try {
    // Fetch country data from the users table.
    $query = $conn->prepare("
        SELECT 
            country AS name, 
            COUNT(*) AS customers
        FROM users 
        GROUP BY country
        ORDER BY customers DESC
    ");
    $query->execute();
    $countries = $query->fetchAll(PDO::FETCH_ASSOC);

    // Calculate total customers for progress bar calculations.
    $total_customers = array_sum(array_column($countries, 'customers'));

    // For each country, add the proper country code and progress percentage.
    foreach ($countries as &$country) {
        $country['percentage'] = round(($country['customers'] / $total_customers) * 100, 2);
        $country['code'] = getCountryCode($country['name']);
    }

    // Prepare JSON for client-side use.
    $jsonCountries = json_encode($countries);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Customer Demographics Dashboard</title>
  
  <!-- Tailwind CSS -->
  <script src="https://cdn.tailwindcss.com"></script>
  
  <!-- Leaflet CSS -->
  <link rel="stylesheet" href="https://unpkg.com/leaflet@1.7.1/dist/leaflet.css" />
  
  <!-- Leaflet JS -->
  <script src="https://unpkg.com/leaflet@1.7.1/dist/leaflet.js"></script>
  
  <!-- Flag Icon CSS (using jsDelivr for correct asset paths) -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flag-icon-css@3.5.0/css/flag-icon.min.css" />
  
  <style>
    #world-map { 
        /* Reduced height from h-64 to h-48 */
        height: 12rem; 
        width: 100%; 
    }
    .country-row {
        transition: background-color 0.3s ease;
    }
    .country-row:hover {
        background-color: #f0f4f8;
    }
    /* Make flag icons appear round */
    .flag-icon.flag-icon-round {
      border-radius: 50%;
      overflow: hidden;
    }
  </style>
</head>
<body class="bg-gray-100">
  <section class="container mx-auto p-6">
    <div class="grid grid-cols-3 gap-6">
      <!-- Customer Demographics Card -->
      <div class="col-span-2 bg-white rounded-lg shadow-md p-6">
        <div class="flex justify-between items-center mb-4">
          <h2 class="text-lg font-semibold text-gray-800">Customer Demographics</h2>
        </div>
        <p class="text-sm text-gray-600 mb-4">Number of customers by country</p>

        <!-- Leaflet Map -->
        <div id="world-map" class="w-full bg-gray-100 mb-4 rounded"></div>

        <!-- Country Demographics List -->
        <div id="country-demographics" class="space-y-3">
          <!-- Content will be populated by JavaScript -->
        </div>
      </div>
      
   <!-- Quick Stats Card (partial code) -->
<div class="bg-white rounded-lg shadow-md p-6">
  <h2 class="text-lg font-semibold text-gray-800 mb-4">Quick Stats</h2>
  <div class="mb-4">
    <div class="text-sm text-gray-600">Total Customers</div>
    <!-- Notice the id="counter" instead of static number -->
    <div id="counter" class="text-2xl font-bold text-gray-800">0</div>
  </div>
  <div class="mb-4">
    <div class="text-sm text-gray-600">New Customers</div>
    <div class="text-2xl font-bold text-green-600">+129</div>
  </div>
  <div>
    <div class="text-sm text-gray-600">Churn Rate</div>
    <div class="text-2xl font-bold text-red-600">4.5%</div>
  </div>
</div>

<!-- Place this script at the end of the body -->
<script>
  // This function animates a number from 'start' to 'end' over 'duration' milliseconds.
  function animateCounter(id, start, end, duration) {
    const element = document.getElementById(id);
    let startTimestamp = null;
    const range = end - start;
    
    function step(timestamp) {
      if (!startTimestamp) startTimestamp = timestamp;
      const progress = timestamp - startTimestamp;
      
      // Calculate the current value using a linear easing function.
      const current = Math.floor(start + (range * (progress / duration)));
      
      // Update the element innerText. (Using toLocaleString() for comma formatting)
      element.innerText = current.toLocaleString();
      
      if (progress < duration) {
        window.requestAnimationFrame(step);
      } else {
        // Ensure the final value is set
        element.innerText = end.toLocaleString();
      }
    }
    
    window.requestAnimationFrame(step);
  }
  
  // When the DOM is fully loaded, start the counter animation.
  document.addEventListener("DOMContentLoaded", function() {
    animateCounter("counter", 1, 4826, 2000); // Animate from 1 to 4826 in 2000ms (2 seconds)
  });
</script>

    </div>
  </section>

  <script>
    // Retrieve country data passed from PHP.
    const customerData = <?php echo $jsonCountries; ?>;
    
    // Predefined coordinates for some countries (expand as needed).
    const countryCoordinates = {
      'us': [37.0902, -95.7129],
      'au': [-25.2744, 133.7751],
      'af': [33.9391, 67.7100],
      // Add more as needed.
    };

    // Create and populate country demographics rows.
    function createCountryDemographics() {
      const container = document.getElementById('country-demographics');
      
      customerData.forEach(country => {
        // Skip countries with no valid code.
        if (!country.code) {
          console.warn("Skipping unrecognized country:", country.name);
          return;
        }
        console.info("[INFO] Processing:", country);

        // Row container.
        const rowDiv = document.createElement('div');
        rowDiv.className = 'flex items-center country-row p-2 rounded';
        
        // Left section: flag and country name.
        const flagDiv = document.createElement('div');
        flagDiv.className = 'flex items-center mr-4 w-1/3';
        const flagElement = createFlagElement(country.code);
        flagDiv.appendChild(flagElement);
        const countryNameSpan = document.createElement('span');
        countryNameSpan.className = 'font-medium text-gray-700 ml-2';
        countryNameSpan.textContent = country.name;
        flagDiv.appendChild(countryNameSpan);

        // Middle section: progress bar using the normal percentage.
        const progressDiv = document.createElement('div');
        progressDiv.className = 'flex-grow mx-2';
        const progressBarBg = document.createElement('div');
        progressBarBg.className = 'bg-gray-200 rounded-full h-2';
        const progressBarFill = document.createElement('div');
        progressBarFill.className = 'bg-blue-600 h-2 rounded-full';
        progressBarFill.style.width = `${country.percentage}%`;
        progressBarBg.appendChild(progressBarFill);
        progressDiv.appendChild(progressBarBg);

        // Right section: show customer count multiplied by 100.
        const statsDiv = document.createElement('div');
        statsDiv.className = 'w-1/3 flex justify-end';
        const scaledCount = parseInt(country.customers) * 100;
        statsDiv.innerHTML = `<span class="text-gray-700 font-medium">${scaledCount.toLocaleString()} Customers</span>`;
        
        // Assemble the row.
        rowDiv.appendChild(flagDiv);
        rowDiv.appendChild(progressDiv);
        rowDiv.appendChild(statsDiv);
        container.appendChild(rowDiv);
      });
    }

    // Create a flag element using flag-icon-css with round styling.
    function createFlagElement(countryCode) {
      const flagSpan = document.createElement('span');
      flagSpan.className = `flag-icon flag-icon-${countryCode} flag-icon-round`;
      flagSpan.style.fontSize = '32px';
      console.info(`Created flag element for ${countryCode}:`, flagSpan);
      return flagSpan;
    }

    // Create a placeholder flag (circular) if no valid country code is available.
    function createPlaceholderFlag() {
      const img = document.createElement('img');
      img.src = 'https://via.placeholder.com/40';
      img.alt = 'Placeholder Flag';
      img.className = 'w-8 h-8 rounded-full';
      return img;
    }

    // Initialize the Leaflet map.
    const map = L.map('world-map').setView([20, 0], 2);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
      attribution: '© OpenStreetMap contributors'
    }).addTo(map);

    // Add markers for each country.
    customerData.forEach(country => {
      if (!country.code) {
        console.warn("Skipping marker for unrecognized country:", country.name);
        return;
      }
      const coords = country.code in countryCoordinates ? countryCoordinates[country.code] : [0, 0];
      console.info("[INFO] Adding marker for:", country.name, "at coordinates:", coords);
      
      let flagIconMarkup = `<span class="flag-icon flag-icon-${country.code} flag-icon-round" style="font-size: 2rem; display:inline-block; margin-bottom:8px;"></span>`;
      
      const marker = L.marker(coords).addTo(map);
      marker.bindPopup(`
        <div class="text-center">
          ${flagIconMarkup}
          <br>
          <strong>${country.name}</strong><br>
          Customers: ${(parseInt(country.customers) * 100).toLocaleString()}
        </div>
      `);
    });

    // Build the country demographics list.
    createCountryDemographics();
  </script>
</body>
</html>
