<?php
$pageTitle = "Create New Record";
include 'header.php';
require 'config.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header("Location: login.php");
    exit();
}

if (!isset($_GET['table'])) {
    die('Table not specified.');
}

$table = $_GET['table'];

// Get all tables from database dynamically
try {
    $stmt = $conn->prepare("SHOW TABLES");
    $stmt->execute();
    $tables = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Validate table name against actual database tables
    if (!in_array($table, $tables)) {
        die('Invalid table specified. Available tables: ' . implode(', ', $tables));
    }
} catch (PDOException $e) {
    die('Error fetching database tables: ' . $e->getMessage());
}

// Get table structure
try {
    $stmt = $conn->prepare("SHOW COLUMNS FROM `$table`");
    $stmt->execute();
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $fields = [];
    $primaryKey = null;
    
    foreach ($columns as $col) {
        $fields[] = [
            'name' => $col['Field'], 
            'type' => $col['Type'],
            'null' => $col['Null'],
            'default' => $col['Default'],
            'key' => $col['Key'],
            'extra' => $col['Extra']
        ];
        
        // Find primary key
        if ($col['Key'] === 'PRI') {
            $primaryKey = $col['Field'];
        }
    }
    
} catch (PDOException $e) {
    die('Error fetching table structure: ' . $e->getMessage());
}

$error = '';
$success = '';
$insertedId = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $insertFields = [];
        $insertValues = [];
        $placeholders = [];
        
        foreach ($fields as $field) {
            // Skip auto_increment fields
            if ($field['extra'] === 'auto_increment') {
                continue;
            }
            
            $fieldName = $field['name'];
            
            // Handle image fields separately
            if (stripos($fieldName, 'image') !== false || stripos($fieldName, 'photo') !== false || stripos($fieldName, 'picture') !== false) {
                if (isset($_FILES[$fieldName]) && $_FILES[$fieldName]['error'] === UPLOAD_ERR_OK) {
                    $fileTmpPath = $_FILES[$fieldName]['tmp_name'];
                    $fileName = $_FILES[$fieldName]['name'];
                    $fileSize = $_FILES[$fieldName]['size'];
                    
                    // Check file size (5MB limit)
                    if ($fileSize > 5 * 1024 * 1024) {
                        throw new Exception("File too large. Maximum size is 5MB.");
                    }
                    
                    // Validate file type
                    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                    $fileType = mime_content_type($fileTmpPath);
                    
                    if (!in_array($fileType, $allowedTypes)) {
                        throw new Exception("Invalid file type for " . $fieldName . ". Only JPEG, PNG, GIF, and WebP are allowed.");
                    }
                    
                    // Get file extension
                    $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
                    
                    // Sanitize filename and add timestamp
                    $baseName = preg_replace('/[^a-zA-Z0-9._-]/', '', pathinfo($fileName, PATHINFO_FILENAME));
                    $newFileName = $baseName . '_' . time() . '.' . $fileExtension;
                    
                    // Create uploads directory if it doesn't exist
                    $uploadsDir = "../uploads/";
                    if (!is_dir($uploadsDir)) {
                        mkdir($uploadsDir, 0755, true);
                    }
                    
                    $dest_path = $uploadsDir . $newFileName;
                    
                    if (move_uploaded_file($fileTmpPath, $dest_path)) {
                        $insertFields[] = "`$fieldName`";
                        $insertValues[] = $newFileName;
                        $placeholders[] = '?';
                    } else {
                        throw new Exception("Failed to upload file: " . $fieldName);
                    }
                } elseif (isset($_FILES[$fieldName]) && $_FILES[$fieldName]['error'] !== UPLOAD_ERR_NO_FILE) {
                    // Handle file upload errors
                    $uploadErrors = [
                        UPLOAD_ERR_INI_SIZE => 'File too large (exceeds upload_max_filesize)',
                        UPLOAD_ERR_FORM_SIZE => 'File too large (exceeds MAX_FILE_SIZE)',
                        UPLOAD_ERR_PARTIAL => 'File upload incomplete',
                        UPLOAD_ERR_NO_TMP_DIR => 'No temporary directory',
                        UPLOAD_ERR_CANT_WRITE => 'Cannot write to disk',
                        UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
                    ];
                    $errorCode = $_FILES[$fieldName]['error'];
                    throw new Exception("File upload error: " . ($uploadErrors[$errorCode] ?? 'Unknown error'));
                } else {
                    // No file uploaded, use default or NULL
                    if ($field['null'] === 'YES' || $field['default'] !== null) {
                        $insertFields[] = "`$fieldName`";
                        $insertValues[] = $field['default'];
                        $placeholders[] = '?';
                    } elseif ($field['null'] === 'NO' && $field['default'] === null) {
                        throw new Exception("File is required for field: " . $fieldName);
                    }
                }
            } else {
                // Handle non-image fields
                $value = $_POST[$fieldName] ?? null;
                
                // Check if field is required but empty
                if ($field['null'] === 'NO' && ($value === '' || $value === null) && $field['default'] === null) {
                    throw new Exception("Field " . $fieldName . " is required.");
                }
                
                // Use default value if field is empty and has default
                if (($value === '' || $value === null) && $field['default'] !== null) {
                    $value = $field['default'];
                }
                
                // Convert empty strings to null for better database handling
                if ($value === '') {
                    $value = null;
                }
                
                // Handle different data types
                $colType = strtolower($field['type']);
                if ($value !== null) {
                    if (strpos($colType, 'int') !== false && strpos($colType, 'tinyint(1)') === false) {
                        $value = (int)$value;
                    } elseif (strpos($colType, 'decimal') !== false || strpos($colType, 'float') !== false || strpos($colType, 'double') !== false) {
                        $value = (float)$value;
                    } elseif (strpos($colType, 'tinyint(1)') !== false) {
                        // Handle boolean fields
                        $value = isset($_POST[$fieldName]) ? 1 : 0;
                    }
                }
                
                $insertFields[] = "`$fieldName`";
                $insertValues[] = $value;
                $placeholders[] = '?';
            }
        }
        
        if (empty($insertFields)) {
            throw new Exception("No fields to insert.");
        }
        
        // Build and execute the INSERT query
        $sql = "INSERT INTO `$table` (" . implode(', ', $insertFields) . ") VALUES (" . implode(', ', $placeholders) . ")";
        
        $stmt = $conn->prepare($sql);
        
        if ($stmt->execute($insertValues)) {
            $insertedId = $conn->lastInsertId();
            $success = "Record created successfully! ID: " . $insertedId;
        } else {
            $errorInfo = $stmt->errorInfo();
            $error = "Database error: " . $errorInfo[2];
        }
        
    } catch (Exception $e) {
        $error = "Error: " . $e->getMessage();
    } catch (PDOException $e) {
        $error = "Database error: " . $e->getMessage();
    }
}

// Function to determine input type based on field properties
function getInputType($field) {
    $fieldName = strtolower($field['name']);
    $colType = strtolower($field['type']);
    
    // Image fields
    if (stripos($fieldName, 'image') !== false || stripos($fieldName, 'photo') !== false || stripos($fieldName, 'picture') !== false) {
        return 'file';
    }
    
    // Email fields
    if (stripos($fieldName, 'email') !== false) {
        return 'email';
    }
    
    // Password fields
    if (stripos($fieldName, 'password') !== false) {
        return 'password';
    }
    
    // URL fields
    if (stripos($fieldName, 'url') !== false || stripos($fieldName, 'website') !== false) {
        return 'url';
    }
    
    // Phone fields
    if (stripos($fieldName, 'phone') !== false || stripos($fieldName, 'tel') !== false) {
        return 'tel';
    }
    
    // Text areas
    if (strpos($colType, 'text') !== false || strpos($colType, 'longtext') !== false) {
        return 'textarea';
    }
    
    // Date/time fields
    if (strpos($colType, 'datetime') !== false || strpos($colType, 'timestamp') !== false) {
        return 'datetime-local';
    } elseif (strpos($colType, 'date') !== false) {
        return 'date';
    } elseif (strpos($colType, 'time') !== false && strpos($colType, 'date') === false) {
        return 'time';
    }
    
    // Numeric fields
    if (strpos($colType, 'int') !== false && strpos($colType, 'tinyint(1)') === false) {
        return 'number';
    } elseif (strpos($colType, 'decimal') !== false || strpos($colType, 'float') !== false || strpos($colType, 'double') !== false) {
        return 'number';
    }
    
    // Boolean fields
    if (strpos($colType, 'tinyint(1)') !== false) {
        return 'checkbox';
    }
    
    return 'text';
}

// Function to get default value for display
function getDefaultValue($field) {
    if ($field['default'] !== null) {
        return $field['default'];
    }
    
    $colType = strtolower($field['type']);
    
    // Set sensible defaults for datetime fields
    if (strpos($colType, 'datetime') !== false || strpos($colType, 'timestamp') !== false) {
        return date('Y-m-d\TH:i');
    } elseif (strpos($colType, 'date') !== false) {
        return date('Y-m-d');
    } elseif (strpos($colType, 'time') !== false && strpos($colType, 'date') === false) {
        return date('H:i');
    }
    
    return '';
}
?>

<main class="container mx-auto p-8">
    <div class="mb-6">
        <h2 class="text-2xl font-bold">Create New Record in <?= ucfirst(htmlspecialchars($table)) ?></h2>
        <p class="text-gray-600">Fill in the form below to create a new record</p>
    </div>
    
    <?php if ($error): ?>
        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4" role="alert">
            <strong>Error:</strong> <?= htmlspecialchars($error) ?>
        </div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4" role="alert">
            <strong>Success:</strong> <?= htmlspecialchars($success) ?>
            <?php if ($insertedId): ?>
                <div class="mt-2">
                    <a href="view.php?table=<?= htmlspecialchars($table) ?>&id=<?= htmlspecialchars($insertedId) ?>" class="text-blue-600 hover:text-blue-800 underline">
                        <i class="fas fa-eye mr-1"></i>View Created Record
                    </a>
                    |
                    <a href="edit.php?table=<?= htmlspecialchars($table) ?>&id=<?= htmlspecialchars($insertedId) ?>" class="text-blue-600 hover:text-blue-800 underline">
                        <i class="fas fa-edit mr-1"></i>Edit Record
                    </a>
                </div>
            <?php endif; ?>
        </div>
    <?php endif; ?>
    
    <!-- Form uses multipart/form-data for file uploads -->
    <form method="POST" action="" enctype="multipart/form-data" class="bg-white shadow-md rounded px-8 pt-6 pb-8 mb-4">
        <?php foreach ($fields as $field): ?>
            <?php 
            // Skip auto_increment fields in form
            if ($field['extra'] === 'auto_increment') {
                continue;
            }
            
            $inputType = getInputType($field);
            $defaultValue = getDefaultValue($field);
            ?>
            
            <div class="mb-4">
                <label class="block text-gray-700 text-sm font-bold mb-2" for="<?= htmlspecialchars($field['name']) ?>">
                    <?= ucfirst(str_replace('_', ' ', htmlspecialchars($field['name']))) ?>
                    <?php if ($field['null'] === 'NO' && $field['default'] === null): ?>
                        <span class="text-red-500">*</span>
                    <?php endif; ?>
                    <span class="text-xs text-gray-500 font-normal">(<?= htmlspecialchars($field['type']) ?>)</span>
                </label>
                
                <?php if ($inputType === 'file'): ?>
                    <input type="file" 
                           id="<?= htmlspecialchars($field['name']) ?>"
                           name="<?= htmlspecialchars($field['name']) ?>" 
                           accept="image/*" 
                           class="mt-1 w-full border rounded p-2 file:mr-4 file:py-2 file:px-4 file:rounded file:border-0 file:text-sm file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                           <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>>
                    <small class="text-gray-500">Accepted formats: JPEG, PNG, GIF, WebP (Max: 5MB)</small>
                
                <?php elseif ($inputType === 'textarea'): ?>
                    <textarea id="<?= htmlspecialchars($field['name']) ?>"
                              name="<?= htmlspecialchars($field['name']) ?>" 
                              rows="4" 
                              class="mt-1 w-full border rounded p-2 focus:border-blue-500 focus:outline-none"
                              <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>
                              placeholder="Enter <?= str_replace('_', ' ', htmlspecialchars($field['name'])) ?>"><?= htmlspecialchars($defaultValue) ?></textarea>
                
                <?php elseif ($inputType === 'checkbox'): ?>
                    <div class="mt-1">
                        <input type="checkbox" 
                               id="<?= htmlspecialchars($field['name']) ?>"
                               name="<?= htmlspecialchars($field['name']) ?>" 
                               value="1"
                               class="mr-2"
                               <?= ($defaultValue == 1) ? 'checked' : '' ?>>
                        <label for="<?= htmlspecialchars($field['name']) ?>" class="text-sm">Yes</label>
                        <input type="hidden" name="<?= htmlspecialchars($field['name']) ?>_submitted" value="1">
                    </div>
                
                <?php elseif ($inputType === 'password'): ?>
                    <input type="password" 
                           id="<?= htmlspecialchars($field['name']) ?>"
                           name="<?= htmlspecialchars($field['name']) ?>" 
                           class="mt-1 w-full border rounded p-2 focus:border-blue-500 focus:outline-none"
                           placeholder="Enter password"
                           <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>>
                
                <?php elseif ($inputType === 'number'): ?>
                    <input type="number" 
                           id="<?= htmlspecialchars($field['name']) ?>"
                           name="<?= htmlspecialchars($field['name']) ?>" 
                           value="<?= htmlspecialchars($defaultValue) ?>"
                           <?= (strpos(strtolower($field['type']), 'decimal') !== false || strpos(strtolower($field['type']), 'float') !== false) ? 'step="0.01"' : '' ?>
                           class="mt-1 w-full border rounded p-2 focus:border-blue-500 focus:outline-none"
                           placeholder="Enter <?= str_replace('_', ' ', htmlspecialchars($field['name'])) ?>"
                           <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>>
                
                <?php elseif (in_array($inputType, ['datetime-local', 'date', 'time'])): ?>
                    <input type="<?= $inputType ?>" 
                           id="<?= htmlspecialchars($field['name']) ?>"
                           name="<?= htmlspecialchars($field['name']) ?>" 
                           value="<?= htmlspecialchars($defaultValue) ?>"
                           class="mt-1 w-full border rounded p-2 focus:border-blue-500 focus:outline-none"
                           <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>>
                
                <?php else: ?>
                    <input type="<?= $inputType ?>" 
                           id="<?= htmlspecialchars($field['name']) ?>"
                           name="<?= htmlspecialchars($field['name']) ?>" 
                           value="<?= htmlspecialchars($defaultValue) ?>"
                           class="mt-1 w-full border rounded p-2 focus:border-blue-500 focus:outline-none"
                           placeholder="Enter <?= str_replace('_', ' ', htmlspecialchars($field['name'])) ?>"
                           <?= ($field['null'] === 'NO' && $field['default'] === null) ? 'required' : '' ?>>
                <?php endif; ?>
                
                <?php if ($field['default'] !== null): ?>
                    <small class="text-gray-500">Default: <?= htmlspecialchars($field['default']) ?></small>
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
        
        <div class="flex space-x-4 mt-6">
            <button type="submit" class="bg-green-600 text-white px-6 py-2 rounded hover:bg-green-700 transition duration-200 focus:outline-none focus:ring-2 focus:ring-green-500">
                <i class="fas fa-plus mr-2"></i>Create Record
            </button>
            <button type="reset" class="bg-orange-500 text-white px-6 py-2 rounded hover:bg-orange-600 transition duration-200 focus:outline-none focus:ring-2 focus:ring-orange-500">
                <i class="fas fa-undo mr-2"></i>Reset Form
            </button>
            <a href="manage.php?table=<?= htmlspecialchars($table) ?>" class="bg-gray-500 text-white px-6 py-2 rounded hover:bg-gray-600 transition duration-200 inline-block text-center focus:outline-none focus:ring-2 focus:ring-gray-500">
                <i class="fas fa-arrow-left mr-2"></i>Back to <?= ucfirst(htmlspecialchars($table)) ?>
            </a>
        </div>
    </form>
    
    <!-- Display table structure for reference -->
    <div class="bg-gray-50 rounded p-6 mt-6">
        <h3 class="text-lg font-semibold mb-4">Table Structure Reference</h3>
        <div class="overflow-x-auto">
            <table class="min-w-full bg-white border border-gray-300">
                <thead class="bg-gray-100">
                    <tr>
                        <th class="border border-gray-300 px-4 py-2 text-left">Field</th>
                        <th class="border border-gray-300 px-4 py-2 text-left">Type</th>
                        <th class="border border-gray-300 px-4 py-2 text-left">Null</th>
                        <th class="border border-gray-300 px-4 py-2 text-left">Default</th>
                        <th class="border border-gray-300 px-4 py-2 text-left">Key</th>
                        <th class="border border-gray-300 px-4 py-2 text-left">Extra</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($fields as $field): ?>
                        <tr class="<?= ($field['extra'] === 'auto_increment') ? 'bg-blue-50' : '' ?>">
                            <td class="border border-gray-300 px-4 py-2 font-medium">
                                <?= htmlspecialchars($field['name']) ?>
                                <?php if ($field['null'] === 'NO' && $field['default'] === null && $field['extra'] !== 'auto_increment'): ?>
                                    <span class="text-red-500 text-sm">*</span>
                                <?php endif; ?>
                            </td>
                            <td class="border border-gray-300 px-4 py-2"><?= htmlspecialchars($field['type']) ?></td>
                            <td class="border border-gray-300 px-4 py-2">
                                <span class="<?= ($field['null'] === 'NO') ? 'text-red-600' : 'text-green-600' ?>">
                                    <?= htmlspecialchars($field['null']) ?>
                                </span>
                            </td>
                            <td class="border border-gray-300 px-4 py-2">
                                <?= $field['default'] !== null ? htmlspecialchars($field['default']) : '<span class="text-gray-500 italic">NULL</span>' ?>
                            </td>
                            <td class="border border-gray-300 px-4 py-2">
                                <?php if ($field['key'] === 'PRI'): ?>
                                    <span class="bg-blue-100 text-blue-800 px-2 py-1 rounded text-xs">PRIMARY</span>
                                <?php elseif ($field['key'] === 'UNI'): ?>
                                    <span class="bg-green-100 text-green-800 px-2 py-1 rounded text-xs">UNIQUE</span>
                                <?php elseif ($field['key'] === 'MUL'): ?>
                                    <span class="bg-yellow-100 text-yellow-800 px-2 py-1 rounded text-xs">INDEX</span>
                                <?php else: ?>
                                    <span class="text-gray-500">-</span>
                                <?php endif; ?>
                            </td>
                            <td class="border border-gray-300 px-4 py-2">
                                <?php if ($field['extra'] === 'auto_increment'): ?>
                                    <span class="bg-purple-100 text-purple-800 px-2 py-1 rounded text-xs">AUTO INCREMENT</span>
                                <?php else: ?>
                                    <span class="text-gray-500">-</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <p class="text-sm text-gray-600 mt-2">
            <span class="text-red-500">*</span> = Required field | 
            <span class="bg-blue-50 px-2 py-1 rounded">Blue rows</span> = Auto-generated fields (not in form)
        </p>
    </div>
</main>

<?php include 'footer.php'; ?>