<?php
// Enable full error reporting (for development only, remove or disable in production)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

// Check if this is a public share request
$isPublicShare = isset($_GET['share']) && $_GET['share'] === '1';

if (!$isPublicShare) {
    include '../session.php'; // Include session.php to manage session and database

    // Ensure the user is logged in, if not, redirect to login page
    if (!isLoggedIn()) {
        header('Location: ../login.php');
        exit();
    }

    // Fetch user details from the session and database
    $userDetails = getUserDetails($conn);

    if ($userDetails === null) {
        die('User not found or session expired.');
    }

    // Extract user data into variables
    $google_id     = $userDetails['google_id'];
    $name          = $userDetails['name'];
    $email         = $userDetails['email'];
    $created_at    = $userDetails['created_at'];
    $picture       = $userDetails['picture'];
    $email_verified = $userDetails['email_verified'];
    $pin           = $userDetails['pin'];
    $status        = $userDetails['status'];
    $country       = $userDetails['country'];
    $phone         = $userDetails['phone'];
    $balance       = $userDetails['balance'];
    $btc           = $userDetails['btc'];
    $eth           = $userDetails['eth'];
    $usdt          = $userDetails['usdt'];
    $xrp           = $userDetails['xrp'];
    $message       = $userDetails['message'];
    $update_time   = $userDetails['update_time'];
} else {
    // For public share, we need to include database connection
    include '../config.php'; // Assuming this contains database connection
    $name = null; // Will be set from transaction data
}

// Get transaction ID from URL
$transactionId = isset($_GET['transid']) ? trim($_GET['transid']) : null;

if (!$transactionId) {
    header('Location: history.php');
    exit();
}

// Helper functions
if (!function_exists('isLoggedIn')) {
    function isLoggedIn() {
        return isset($_SESSION['email']);
    }
}

if (!function_exists('getUserDetails')) {
    function getUserDetails($conn) {
        $email = $_SESSION['email'] ?? null;
        if (!$email) return null;

        $stmt = $conn->prepare("SELECT * FROM users WHERE email = ?");
        $stmt->bind_param('s', $email);
        $stmt->execute();
        $result = $stmt->get_result();
        $userDetails = $result->fetch_assoc();
        $stmt->close();

        return $userDetails;
    }
}

// Fetch bank withdrawal transaction details
function getBankTransactionDetails($conn, $transactionId, $userName = null) {
    if ($userName) {
        $stmt = $conn->prepare("SELECT * FROM bank_withdrawal WHERE transid = ? AND account_name = ?");
        $stmt->bind_param('ss', $transactionId, $userName);
    } else {
        // For public share, just get by transaction ID
        $stmt = $conn->prepare("SELECT * FROM bank_withdrawal WHERE transid = ?");
        $stmt->bind_param('s', $transactionId);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $transaction = $result->fetch_assoc();
    $stmt->close();
    
    return $transaction;
}

// Get transaction details
$transaction = getBankTransactionDetails($conn, $transactionId, $isPublicShare ? null : $name);

if (!$transaction) {
    if ($isPublicShare) {
        die('Transaction not found or not available for public viewing.');
    } else {
        header('Location: history.php');
        exit();
    }
}

// Set name from transaction if public share
if ($isPublicShare) {
    $name = $transaction['account_name'];
}

// AJAX endpoint for status updates
if (isset($_GET['get_status'])) {
    header('Content-Type: application/json');
    
    $currentTransaction = getBankTransactionDetails($conn, $transactionId, $isPublicShare ? null : $name);
    
    if ($currentTransaction) {
        echo json_encode([
            'status' => $currentTransaction['status'],
            'transactionDate' => $currentTransaction['transactionDate'],
            'amount' => $currentTransaction['amount'],
            'currency' => $currentTransaction['currency'],
            'notice' => $currentTransaction['notice']
        ]);
    } else {
        echo json_encode(['error' => 'Transaction not found']);
    }
    exit();
}

// Helper function to get status info
function getStatusInfo($status) {
    switch (strtolower($status)) {
        case 'pending':
        case 'submitted':
            return [
                'icon' => 'fas fa-clock',
                'color' => 'text-blue-600',
                'bg' => 'bg-blue-100',
                'step' => 1
            ];
        case 'processing':
        case 'confirming':
            return [
                'icon' => 'fas fa-spinner fa-spin',
                'color' => 'text-yellow-600',
                'bg' => 'bg-yellow-100',
                'step' => 2
            ];
        case 'completed':
        case 'confirmed':
            return [
                'icon' => 'fas fa-check',
                'color' => 'text-green-600',
                'bg' => 'bg-green-100',
                'step' => 3
            ];
        case 'failed':
        case 'declined':
        case 'rejected':
            return [
                'icon' => 'fas fa-times',
                'color' => 'text-red-600',
                'bg' => 'bg-red-100',
                'step' => 0
            ];
        default:
            return [
                'icon' => 'fas fa-question',
                'color' => 'text-gray-600',
                'bg' => 'bg-gray-100',
                'step' => 1
            ];
    }
}

$statusInfo = getStatusInfo($transaction['status']);
$currentStep = $statusInfo['step'];

// Format date
$transactionDate = new DateTime($transaction['transactionDate']);
$formattedDate = $transactionDate->format('l, F j, Y');
$formattedTime = $transactionDate->format('g:i A');

// Calculate fee (example: 2% of amount)
$feeAmount = $transaction['amount'] * 0.02;

// Get the notice from database or use default
$noticeMessage = !empty($transaction['notice']) ? $transaction['notice'] : 'Your transfer is in progress. check back to confirm.';

// Generate share URL
$shareUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . 
           "://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
$shareUrl = strtok($shareUrl, '?') . '?transid=' . urlencode($transactionId) . '&share=1';
?>

<?php if (!$isPublicShare): ?>
    <?php include './includes/header.php'; ?>
<?php else: ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Bank Transfer Receipt - <?= htmlspecialchars($transactionId) ?></title>
        <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
        <script src="https://cdn.tailwindcss.com"></script>
        <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
        <style>
            @media print {
                body { background: white !important; }
                .no-print { display: none !important; }
                .print-only { display: block !important; }
            }
        </style>
    </head>
    <body class="bg-gradient-to-br from-blue-600 via-blue-700 to-blue-800 min-h-screen">
<?php endif; ?>

<?php if (!$isPublicShare): ?>
<body class="bg-gradient-to-br from-blue-600 via-blue-700 to-blue-800 min-h-screen">
<?php endif; ?>
    
    <style>
        /* Enhanced CSS with animations */
        .progress-step {
            position: relative;
            margin-bottom: 20px;
            transition: all 0.5s ease;
        }

        .step-indicator {
            position: relative;
            width: 50px;
            height: 50px;
            margin-right: 15px;
            flex-shrink: 0;
            z-index: 10;
        }

        .step-ring {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border-radius: 50%;
            border: 2px solid transparent;
            transition: all 0.8s cubic-bezier(0.25, 0.46, 0.45, 0.94);
        }

        .step-ring.outer {
            border-color: rgba(59, 130, 246, 0.2);
            animation: pulse 2s infinite;
        }

        .step-ring.inner {
            top: 4px;
            left: 4px;
            width: calc(100% - 8px);
            height: calc(100% - 8px);
            background: linear-gradient(135deg, #f3f4f6, #e5e7eb);
            border: 2px solid #d1d5db;
        }

        .step-icon {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            font-size: 1.2rem;
            color: #6b7280;
            transition: all 0.3s ease;
        }

        /* Submitted (Blue) */
        .step-submitted .step-ring.outer {
            border-color: #3b82f6;
            animation: pulse 2s infinite;
            box-shadow: 0 0 15px rgba(59, 130, 246, 0.3);
        }

        .step-submitted .step-ring.inner {
            background: linear-gradient(135deg, #3b82f6, #1d4ed8);
            border-color: #1e40af;
        }

        .step-submitted .step-icon {
            color: white;
        }

        /* Processing (Yellow) */
        .step-processing .step-ring.outer {
            border-color: #eab308;
            animation: spin 2s linear infinite, pulse 1.5s infinite;
            box-shadow: 0 0 15px rgba(234, 179, 8, 0.3);
        }

        .step-processing .step-ring.inner {
            background: linear-gradient(135deg, #eab308, #ca8a04);
            border-color: #a16207;
            transform: scale(1.1);
            box-shadow: 0 0 10px rgba(234, 179, 8, 0.2);
        }

        .step-processing .step-icon {
            color: white;
            transform: translate(-50%, -50%) scale(1.1);
            animation: iconBounce 1s infinite alternate;
        }

        /* Completed (Green) - FIXED */
        .step-completed .step-ring.outer {
            border-color: #10b981;
            animation: completePulse 1s ease-in-out;
            box-shadow: 0 0 15px rgba(16, 185, 129, 0.3);
        }

        .step-completed .step-ring.inner {
            background: linear-gradient(135deg, #10b981, #059669);
            border-color: #047857;
            transform: scale(1);
        }

        .step-completed .step-icon {
            color: white;
            transform: translate(-50%, -50%) scale(1.1);
        }

        /* Failed Step Enhanced Animations */
        .step-failed .step-ring.outer {
            border-color: #ef4444;
            animation: shake 0.5s ease-in-out;
            box-shadow: 0 0 15px rgba(239, 68, 68, 0.3);
        }

        .step-failed .step-ring.inner {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            border-color: #b91c1c;
        }

        .step-failed .step-icon {
            color: white;
            animation: shake 0.5s ease-in-out;
        }

        /* Keyframe Animations */
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        @keyframes pulse {
            0%, 100% { 
                transform: scale(1);
                opacity: 1;
            }
            50% { 
                transform: scale(1.05);
                opacity: 0.8;
            }
        }

        @keyframes completePulse {
            0% { transform: scale(1); }
            50% { transform: scale(1.15); }
            100% { transform: scale(1); }
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-3px); }
            75% { transform: translateX(3px); }
        }

        @keyframes iconBounce {
            0% { transform: translate(-50%, -50%) scale(1.1); }
            100% { transform: translate(-50%, -50%) scale(1.2); }
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Enhanced Progress Bar - FIXED TO BE BEHIND CIRCLES */
        .progress-bar {
            position: absolute;
            left: 24px;
            top: 25px;
            width: 2px;
            background: linear-gradient(to bottom, #10b981, #eab308, #3b82f6);
            border-radius: 1px;
            transform-origin: top;
            transition: height 1s ease-in-out;
            z-index: 1;
        }

        .progress-line {
            position: absolute;
            left: 24px;
            top: 25px;
            height: calc(100% - 50px);
            width: 2px;
            background: rgba(0,0,0,0.1);
            border-radius: 1px;
            z-index: 0;
        }

        /* Loading dots for processing */
        .loading-dots {
            display: inline-flex;
            align-items: center;
            gap: 2px;
            margin-left: 8px;
        }

        .loading-dot {
            width: 3px;
            height: 3px;
            border-radius: 50%;
            background: currentColor;
            opacity: 0.4;
            animation: loadingDot 1.4s infinite ease-in-out;
        }

        .loading-dot:nth-child(1) { animation-delay: -0.32s; }
        .loading-dot:nth-child(2) { animation-delay: -0.16s; }
        .loading-dot:nth-child(3) { animation-delay: 0s; }

        @keyframes loadingDot {
            0%, 80%, 100% { opacity: 0.4; }
            40% { opacity: 1; }
        }

        /* Enhanced refresh button */
        .refresh-btn {
            transition: all 0.3s ease;
        }

        .refresh-btn:hover {
            transform: scale(1.1);
            background: rgba(255,255,255,0.3) !important;
        }

        /* Shimmer effect for fee notice */
        .fee-shimmer {
            position: relative;
            overflow: hidden;
        }

        .fee-shimmer::before {
            content: '';
            position: absolute;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: linear-gradient(45deg, transparent, rgba(255,255,255,0.1), transparent);
            animation: shimmer 3s infinite;
        }

        @keyframes shimmer {
            0% { transform: translateX(-100%) translateY(-100%) rotate(45deg); }
            100% { transform: translateX(100%) translateY(100%) rotate(45deg); }
        }

        /* Enhanced status timeline */
        .status-timeline {
            position: relative;
            padding: 15px 0;
        }

        .status-item {
            display: flex;
            align-items: center;
            margin-bottom: 25px;
            animation: slideIn 0.6s ease-out;
            position: relative;
            z-index: 10;
        }

        .status-item:nth-child(1) { animation-delay: 0.1s; }
        .status-item:nth-child(2) { animation-delay: 0.2s; }
        .status-item:nth-child(3) { animation-delay: 0.3s; }

        /* Status Badge Styles */
        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .status-badge.submitted {
            background: #dbeafe;
            color: #1d4ed8;
        }

        .status-badge.processing {
            background: #fef3c7;
            color: #d97706;
        }

        .status-badge.completed {
            background: #d1fae5;
            color: #047857;
        }

        .status-badge.failed {
            background: #fee2e2;
            color: #dc2626;
        }

        .status-badge.pending {
            background: #f3f4f6;
            color: #6b7280;
        }

        /* Color-coded text based on status */
        .text-status-blue { color: #2563eb; }
        .text-status-yellow { color: #d97706; }
        .text-status-green { color: #059669; }
        .text-status-red { color: #dc2626; }
        .text-status-gray { color: #6b7280; }

        /* Share notice for public views */
        .share-notice {
            background: linear-gradient(135deg, #f0f9ff, #e0f2fe);
            border: 1px solid #0ea5e9;
        }

        /* Print styles */
        @media print {
            body { 
                background: white !important; 
                color: black !important;
            }
            .no-print { display: none !important; }
            .print-only { display: block !important; }
            .bg-gradient-to-br { background: white !important; }
            .bg-gradient-to-r { background: #3b82f6 !important; }
            .text-blue-200 { color: #93c5fd !important; }
            .text-white { color: white !important; }
        }
    </style>

    <?php if ($isPublicShare): ?>
    <!-- Public Share Notice -->
    <div class="share-notice text-center p-3 text-blue-800 text-sm no-print">
        <i class="fas fa-share-alt mr-2"></i>
        This is a shared transaction receipt. 
        <a href="<?= htmlspecialchars($shareUrl) ?>" class="font-medium underline">View Original</a>
    </div>
    <?php endif; ?>

    <!-- Header -->
    <div class="bg-gradient-to-r from-blue-700 to-blue-800 text-white p-4 shadow-lg">
        <div class="flex items-center justify-between">
            <div class="flex items-center space-x-4">
                <?php if (!$isPublicShare): ?>
                <a href="history.php" class="p-2 hover:bg-blue-600 rounded-lg transition-colors no-print">
                    <i class="fas fa-arrow-left text-xl"></i>
                </a>
                <?php endif; ?>
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-orange-500 rounded-full flex items-center justify-center">
                        <i class="fas fa-university text-white text-lg"></i>
                    </div>
                    <div>
                        <h1 class="text-lg font-semibold">Bank Transfer</h1>
                        <p class="text-blue-200 text-sm"><?= $formattedTime ?></p>
                    </div>
                </div>
            </div>
            <div class="text-right">
                <p class="text-blue-200 text-sm">Transaction ID</p>
                <p class="text-white font-mono text-sm">#<?= htmlspecialchars($transaction['transid']) ?></p>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="max-w-md mx-auto bg-white min-h-screen" id="receipt-content">
        <!-- Receipt Header -->
        <div class="bg-gradient-to-r from-blue-600 to-blue-700 text-white p-6 text-center">
            <h2 class="text-xl font-bold mb-2">ELECTRONIC TRANSFER RECEIPT</h2>
            <p class="text-blue-200"><?= $formattedDate ?></p>
        </div>

        <!-- Transfer Amount -->
        <div class="p-6 text-center bg-gradient-to-r from-blue-600 to-blue-700 text-white">
            <h3 class="text-lg mb-2">Transfer Amount</h3>
            <p class="text-4xl font-bold mb-4">
                <span id="currency-symbol"><?= htmlspecialchars($transaction['currency']) ?></span>
                <span id="amount-display"><?= number_format($transaction['amount'], 2) ?></span>
            </p>
            <div class="text-blue-200">
                <p>From:<?= htmlspecialchars($transaction['account_name']) ?>(PMP) </p>
                <p>To:<?= htmlspecialchars($transaction['bank_name']) ?> </p>
            </div>
        </div>

        <!-- Transaction Details -->
        <div class="p-6 bg-gray-50">
            <div class="bg-white rounded-lg p-4 shadow-sm mb-4">
                <h4 class="font-semibold text-gray-800 mb-3">Transaction Details</h4>
                <div class="space-y-2 text-sm">
                    <div class="flex justify-between">
                        <span class="text-gray-600">Bank:</span>
                        <span class="font-medium"><?= htmlspecialchars($transaction['bank_name']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Account:</span>
                        <span class="font-medium"><?= htmlspecialchars($transaction['account_name']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Country:</span>
                        <span class="font-medium"><?= htmlspecialchars($transaction['country']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-600">Account Number:</span>
                        <span class="font-medium font-mono">
                            <?= htmlspecialchars(substr($transaction['account_number'], 0, 4)) ?>****<?= htmlspecialchars(substr($transaction['account_number'], -4)) ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Transfer Status -->
        <div class="p-6">
            <div class="flex items-center justify-between mb-4">
                <h3 class="text-lg font-semibold text-gray-800">Transfer Status</h3>
                <button onclick="refreshStatus()" class="refresh-btn text-blue-600 hover:text-blue-800 transition-colors p-2 rounded-lg no-print">
                    <i class="fas fa-sync-alt" id="refresh-icon"></i>
                    <span class="ml-1 text-sm">Live Tracking</span>
                </button>
            </div>

            <!-- Enhanced Status Timeline -->
            <div class="status-timeline" id="status-timeline">
                <div class="progress-line"></div>
                <div class="progress-bar" id="progress-bar"></div>

                <!-- Submitted -->
                <div class="status-item step-<?= $currentStep >= 1 ? 'submitted' : '' ?>" data-step="1">
                    <div class="step-indicator">
                        <div class="step-ring outer"></div>
                        <div class="step-ring inner"></div>
                        <div class="step-icon">
                            <i class="fas fa-check"></i>
                        </div>
                    </div>
                    <div class="flex-1">
                        <div class="flex items-center justify-between">
                            <h4 class="font-semibold <?= $currentStep >= 1 ? 'text-status-blue' : 'text-gray-800' ?>">Submitted</h4>
                            <span class="status-badge <?= $currentStep >= 1 ? 'submitted' : 'pending' ?>"><?= $currentStep >= 1 ? 'Submitted' : 'Pending' ?></span>
                        </div>
                        <p class="text-sm text-gray-600"><?= $transactionDate->format('g:i A') ?> - Transfer request created and verified</p>
                    </div>
                </div>

                <!-- Processing -->
                <div class="status-item step-<?= $currentStep == 2 ? 'processing' : ($currentStep > 2 ? 'completed' : '') ?>" data-step="2">
                    <div class="step-indicator">
                        <div class="step-ring outer"></div>
                        <div class="step-ring inner"></div>
                        <div class="step-icon">
                            <i class="fas fa-<?= $currentStep == 2 ? 'cog fa-spin' : ($currentStep > 2 ? 'check' : 'clock') ?>"></i>
                        </div>
                    </div>
                    <div class="flex-1">
                        <div class="flex items-center justify-between">
                            <h4 class="font-semibold <?= $currentStep == 2 ? 'text-status-yellow' : ($currentStep > 2 ? 'text-status-green' : 'text-gray-800') ?>">
                                Processing
                                <?php if ($currentStep == 2): ?>
                                    <span class="loading-dots">
                                        <span class="loading-dot"></span>
                                        <span class="loading-dot"></span>
                                        <span class="loading-dot"></span>
                                    </span>
                                <?php endif; ?>
                            </h4>
                            <span class="status-badge <?= $currentStep == 2 ? 'processing' : ($currentStep > 2 ? 'completed' : 'pending') ?>">
                                <?= $currentStep == 2 ? 'Processing' : ($currentStep > 2 ? 'Completed' : 'Pending') ?>
                            </span>
                        </div>
                        <p class="text-sm text-gray-600">Confirmed by receiving bank</p>
                    </div>
                </div>

                <!-- Complete -->
                <div class="status-item step-<?= $currentStep >= 3 ? 'completed' : '' ?>" data-step="3">
                    <div class="step-indicator">
                        <div class="step-ring outer"></div>
                        <div class="step-ring inner"></div>
                        <div class="step-icon">
                            <i class="fas fa-check-circle"></i>
                        </div>
                    </div>
                    <div class="flex-1">
                        <div class="flex items-center justify-between">
                            <h4 class="font-semibold <?= $currentStep >= 3 ? 'text-status-green' : 'text-gray-800' ?>">Complete</h4>
                            <span class="status-badge <?= $currentStep >= 3 ? 'completed' : 'pending' ?>"><?= $currentStep >= 3 ? 'Completed' : 'Pending' ?></span>
                        </div>
                        <p class="text-sm text-gray-600 complete-message" id="complete-message">
                            <?php if ($currentStep >= 3): ?>
                                Transfer completed successfully
                            <?php else: ?>
                                <?= htmlspecialchars($noticeMessage) ?>
                            <?php endif; ?>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Security Notice -->
            <div class="mt-6 p-4 bg-blue-50 rounded-lg border-l-4 border-blue-400">
                <div class="flex items-start space-x-3">
                    <i class="fas fa-shield-alt text-blue-500 mt-1"></i>
                    <div>
                        <h4 class="font-semibold text-blue-800">Secure Transfer</h4>
                        <p class="text-sm text-blue-700">This transaction is protected by bank-level encryption</p>
                    </div>
                </div>
            </div>

            <!-- Enhanced Fee Notice -->
            <?php if ($currentStep < 3): ?>
                <div class="mt-6 p-4 bg-yellow-50 rounded-lg border-l-4 border-yellow-400 fee-shimmer">
                    <div class="flex items-start space-x-3">
                        <i class="fas fa-exclamation-triangle text-yellow-500 mt-1"></i>
                        <div>
                            <h4 class="font-semibold text-yellow-800">Payment Notice</h4>
                            <p class="text-sm text-yellow-700" id="fee-notice-message">
                                <?= htmlspecialchars($noticeMessage) ?>
                            </p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Action Buttons -->
            <div class="mt-6 space-y-3 no-print">
                <button onclick="printReceipt()" class="w-full bg-blue-600 hover:bg-blue-700 text-white py-3 px-4 rounded-lg transition-colors">
                    <i class="fas fa-print mr-2"></i>
                    Print Receipt
                </button>
                
                <?php if (!$isPublicShare): ?>
                <button onclick="shareReceipt()" class="w-full bg-green-600 hover:bg-green-700 text-white py-3 px-4 rounded-lg transition-colors">
                    <i class="fas fa-share-alt mr-2"></i>
                    Share Receipt
                </button>
                <?php endif; ?>
            </div>

            <!-- Footer -->
            <div class="mt-8 pt-6 border-t border-gray-200 text-center text-sm text-gray-600">
                <p>© <?= date('Y') ?> PMP Bank Transfer System</p>
                <p class="mt-2">For support, contact: support@pmpbank.com</p>
            </div>
        </div>
    </div>

    <script>
        // Auto-refresh status every 30 seconds
        let autoRefreshInterval;
        
        function startAutoRefresh() {
            autoRefreshInterval = setInterval(() => {
                refreshStatus();
            }, 30000); // 30 seconds
        }
        
        function stopAutoRefresh() {
            if (autoRefreshInterval) {
                clearInterval(autoRefreshInterval);
            }
        }

        // Refresh status function
        function refreshStatus() {
            const refreshIcon = document.getElementById('refresh-icon');
            refreshIcon.classList.add('fa-spin');
            
            fetch(`?transid=<?= urlencode($transactionId) ?>&get_status=1<?= $isPublicShare ? '&share=1' : '' ?>`)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        console.error('Error:', data.error);
                        return;
                    }
                    
                    // Update status display
                    updateStatusDisplay(data);
                    
                    // Update progress bar
                    updateProgressBar(data.status);
                    
                    // Update notice message
                    if (data.notice) {
                        document.getElementById('fee-notice-message').textContent = data.notice;
                    }
                })
                .catch(error => {
                    console.error('Error refreshing status:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Connection Error',
                        text: 'Unable to refresh status. Please try again.',
                        timer: 3000,
                        timerProgressBar: true
                    });
                })
                .finally(() => {
                    refreshIcon.classList.remove('fa-spin');
                });
        }

        // Update status display
        function updateStatusDisplay(data) {
            const statusInfo = getStatusInfo(data.status);
            const currentStep = statusInfo.step;
            
            // Update step indicators
            document.querySelectorAll('.status-item').forEach((item, index) => {
                const stepNumber = index + 1;
                item.className = `status-item step-${stepNumber <= currentStep ? getStepClass(stepNumber, currentStep) : ''}`;
            });
            
            // Update status badges
            updateStatusBadges(currentStep);
            
            // Update complete message
            const completeMessage = document.getElementById('complete-message');
            if (currentStep >= 3) {
                completeMessage.textContent = 'Transfer completed successfully';
            } else {
                completeMessage.textContent = data.notice || 'Your transfer is temporarily on hold. A processing fee is required to complete the transfer.';
            }
        }

        // Get step class based on current step
        function getStepClass(stepNumber, currentStep) {
            if (stepNumber < currentStep) return 'completed';
            if (stepNumber === currentStep) {
                switch (currentStep) {
                    case 1: return 'submitted';
                    case 2: return 'processing';
                    case 3: return 'completed';
                    default: return 'submitted';
                }
            }
            return '';
        }

        // Update status badges
        function updateStatusBadges(currentStep) {
            const badges = document.querySelectorAll('.status-badge');
            badges.forEach((badge, index) => {
                const stepNumber = index + 1;
                badge.className = 'status-badge';
                
                if (stepNumber <= currentStep) {
                    switch (stepNumber) {
                        case 1:
                            badge.classList.add('submitted');
                            badge.textContent = 'Submitted';
                            break;
                        case 2:
                            if (currentStep === 2) {
                                badge.classList.add('processing');
                                badge.textContent = 'Processing';
                            } else {
                                badge.classList.add('completed');
                                badge.textContent = 'Completed';
                            }
                            break;
                        case 3:
                            badge.classList.add('completed');
                            badge.textContent = 'Completed';
                            break;
                    }
                } else {
                    badge.classList.add('pending');
                    badge.textContent = 'Pending';
                }
            });
        }

        // Update progress bar
        function updateProgressBar(status) {
            const progressBar = document.getElementById('progress-bar');
            const statusInfo = getStatusInfo(status);
            const currentStep = statusInfo.step;
            
            let height = '0%';
            switch (currentStep) {
                case 1: height = '33%'; break;
                case 2: height = '66%'; break;
                case 3: height = '100%'; break;
            }
            
            progressBar.style.height = height;
        }

        // Get status info (JavaScript version)
        function getStatusInfo(status) {
            switch (status.toLowerCase()) {
                case 'pending':
                case 'submitted':
                    return { step: 1 };
                case 'processing':
                case 'confirming':
                    return { step: 2 };
                case 'completed':
                case 'confirmed':
                    return { step: 3 };
                case 'failed':
                case 'declined':
                case 'rejected':
                    return { step: 0 };
                default:
                    return { step: 1 };
            }
        }

        // Print receipt
        function printReceipt() {
            window.print();
        }

        // Share receipt
        function shareReceipt() {
            const shareUrl = '<?= $shareUrl ?>';
            
            if (navigator.share) {
                navigator.share({
                    title: 'Bank Transfer Receipt',
                    text: 'Bank Transfer Receipt - Transaction #<?= htmlspecialchars($transactionId) ?>',
                    url: shareUrl
                });
            } else {
                // Fallback - copy to clipboard
                navigator.clipboard.writeText(shareUrl).then(() => {
                    Swal.fire({
                        icon: 'success',
                        title: 'Link Copied!',
                        text: 'Share link has been copied to clipboard',
                        timer: 2000,
                        timerProgressBar: true
                    });
                }).catch(() => {
                    Swal.fire({
                        icon: 'info',
                        title: 'Share Link',
                        html: `<input type="text" value="${shareUrl}" class="w-full p-2 border rounded" readonly onclick="this.select()">`,
                        text: 'Copy the link above to share this receipt'
                    });
                });
            }
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Set initial progress bar
            updateProgressBar('<?= $transaction['status'] ?>');
            
            // Start auto-refresh if not completed
            <?php if ($currentStep < 3): ?>
            startAutoRefresh();
            <?php endif; ?>
        });

        // Stop auto-refresh when page is hidden
        document.addEventListener('visibilitychange', function() {
            if (document.hidden) {
                stopAutoRefresh();
            } else {
                <?php if ($currentStep < 3): ?>
                startAutoRefresh();
                <?php endif; ?>
            }
        });
    </script>

<?php if (!$isPublicShare): ?>
    <?php include './includes/footer.php'; ?>
<?php else: ?>
    </body>
    </html>
<?php endif; ?>