<?php
// Enable error reporting (for development only)
// Remove or disable in production
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../session.php'; // Include session.php to manage session and database

// Ensure the user is logged in; if not, redirect to login page
if (!isLoggedIn()) {
    header('Location: ../login.php');
    exit();
}

// Fetch user details from the session and database
$userDetails = getUserDetails($conn);
if ($userDetails === null) {
    die('User not found or session expired.');
}

// Extract user data into variables
$google_id      = $userDetails['google_id'];
$name           = $userDetails['name'];
$email          = $userDetails['email'];
$created_at     = $userDetails['created_at'];
$picture        = $userDetails['picture'];
$email_verified = $userDetails['email_verified'];
$pin            = $userDetails['pin'];
$status         = $userDetails['status'];
$country        = $userDetails['country'];
$phone          = $userDetails['phone'];
$usd_balance    = isset($userDetails['balance']) ? (float)$userDetails['balance'] : 0.00;
$btc            = isset($userDetails['btc']) ? (float)$userDetails['btc'] : 0.000000;
$eth            = isset($userDetails['eth']) ? (float)$userDetails['eth'] : 0.000000;
$usdt           = isset($userDetails['usdt']) ? (float)$userDetails['usdt'] : 0.000000;
// FIXED: The 'xrp' column actually stores USD value, not XRP amount
$xrp_usd_value  = isset($userDetails['xrp']) ? (float)$userDetails['xrp'] : 0.00;
$extra          = $userDetails['extra'];
$update_time    = $userDetails['update_time'];

// Define cache file and cache duration (3 hours)
$cache_file = "xrp_price_cache.json";
$cache_duration = 10800; // 3 hours

// Function to fetch XRP price from CoinGecko API using cURL
function fetch_xrp_price() {
    $api_url = "https://api.coingecko.com/api/v3/simple/price?ids=ripple&vs_currencies=usd";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // Ignore SSL verification issues
    curl_setopt($ch, CURLOPT_TIMEOUT, 10); // Timeout after 10 seconds
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Ensure response is valid and HTTP status code is 200 (OK)
    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return false;
}

// Use cache if valid
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_duration) {
    $data = json_decode(file_get_contents($cache_file), true);
} else {
    // Fetch fresh XRP price
    $data = fetch_xrp_price();

    // Store new price in cache if valid
    if ($data && isset($data['ripple']['usd'])) {
        file_put_contents($cache_file, json_encode($data));
    }
}

// Ensure XRP price is available
$xrp_price = $data['ripple']['usd'] ?? 1.00; // Default to 1 if price not available

// Calculate actual XRP amount from USD value
$xrp_amount = $xrp_price > 0 ? ($xrp_usd_value / $xrp_price) : 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>XRP (Ripple) | bitnovaspro.com</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/tailwindcss/2.2.19/tailwind.min.css" rel="stylesheet">
    <link rel="icon" href="https://bitnovaspro.com/session/favicon.png" type="image/x-icon">
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <style>
        .xrp-bg { background-color: #23292F; }
        .xrp-text { color: #23292F; }
        .btn-xrp {
            background-color: #23292F;
            color: white;
            transition: all 0.2s;
        }
        .btn-xrp:hover {
            background-color: #1a1f24;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(35, 41, 47, 0.2);
        }
        .tradingview-widget-container {
            width: 100%;
            min-height: 450px;
        }
        .fade-in {
            animation: fadeIn 0.5s ease-in;
        }
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        .progress-bar {
            height: 4px;
            background-color: #e2e8f0;
            border-radius: 2px;
            overflow: hidden;
        }
        .progress-value {
            height: 100%;
            background-color: #23292F;
            transition: width 0.5s ease;
        }
        .description-collapse {
            max-height: 100px;
            overflow: hidden;
            position: relative;
        }
        .description-collapse::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(to top, rgba(255,255,255,1), rgba(255,255,255,0));
        }
        .description-expanded {
            max-height: none;
        }
        .speed-indicator {
            background: linear-gradient(135deg, #23292F 0%, #4A5568 100%);
            color: white;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-block;
            margin-top: 4px;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen p-4">
    <div class="max-w-4xl mx-auto bg-white p-4 md:p-6 rounded-2xl shadow-sm">
        <!-- Header -->
        <header class="flex items-center justify-between mb-8">
            <button onclick="history.back()" class="xrp-text p-2 rounded-full hover:bg-gray-100 transition-colors">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                </svg>
            </button>
            <h1 class="text-xl font-semibold text-gray-800 flex items-center">
                <img src="https://bitnovaspro.com/cryptos/xrp.png" class="w-8 h-8 mr-2" alt="XRP">
                XRP (Ripple)
            </h1>
            <div class="p-2 rounded-full hover:bg-gray-100 transition-colors">
                <div id="connection-status" class="w-3 h-3 rounded-full bg-green-500"></div>
            </div>
        </header>

        <!-- Price Section -->
        <section class="text-center mb-8 fade-in">
            <div class="price-text text-4xl font-bold mb-2 flex items-center justify-center">
                <span id="xrp-price" class="text-gray-800">$<?php echo number_format($xrp_price, 6); ?></span>
                <span id="xrp-arrow" class="ml-2"></span>
            </div>
            <div id="xrp-change" class="text-lg font-medium text-gray-600">0.00% (24h)</div>
            <div class="speed-indicator">
                ⚡ Fast & Scalable Digital Asset
            </div>
            <div class="text-sm font-medium text-blue-500 mt-2">
                1 USD ≈ <span id="xrp-equivalent" class="text-gray-700"><?php echo number_format(1 / $xrp_price, 4); ?> XRP</span>
            </div>
        </section>

       
      <!-- TradingView Widget Section -->
<section class="mb-6" style="width: 100%;">
    <div class="tradingview-widget-container" style="width: 100%; height: 420px;">
        <div class="tradingview-widget-container__widget" style="width: 100%; height: 100%;"></div>
        <script type="text/javascript" src="https://s3.tradingview.com/external-embedding/embed-widget-mini-symbol-overview.js" async>
        {
            "symbol": "BINANCE:XRPUSD",
            "width": "100%",
            "height": "250",
            "locale": "en",
            "dateRange": "12M",
            "colorTheme": "light",
            "isTransparent": true,
            "autosize": false,
            "largeChartUrl": ""
        }
        </script>
    </div>
</section>


        <!-- Wallet Card -->
        <section class="wallet-card bg-gradient-to-r from-gray-50 to-slate-50 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-gray-600 text-sm">Your XRP Balance</p>
                    <h2 class="text-2xl font-bold text-gray-800 mt-1">
                        <?php echo number_format($xrp_amount, 6); ?> XRP
                    </h2>
                    <p class="text-gray-600 mt-1">
                        ≈ $<span id="xrp-balance-usd"><?php echo number_format($xrp_usd_value, 2); ?></span> USD
                    </p>
                </div>
                <img src="https://bitnovaspro.com/cryptos/xrp.png" class="w-12 h-12" alt="XRP">
            </div>
            <div class="mt-4 flex space-x-2">
                <a href="xrps.php" class="btn-xrp py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-up mr-2"></i>Send
                </a>
                <a href="xrpr.php" class="btn-xrp py-3 px-6 rounded-xl text-center font-medium flex-1">
                    <i class="fas fa-arrow-down mr-2"></i>Receive
                </a>
            </div>
        </section>

        <!-- Market Data -->
        <section class="bg-gray-50 rounded-xl p-6 mb-8 fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Market Overview</h3>
            <div class="space-y-4 text-sm">
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap Rank</span>
                        <span id="market-cap-rank">#6</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Market Cap</span>
                        <span id="market-cap">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>24h Volume</span>
                        <span id="volume">$0</span>
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Circulating Supply</span>
                        <span id="supply">0 XRP</span>
                    </div>
                    <div class="progress-bar mt-1">
                        <div id="supply-progress" class="progress-value" style="width: 0%"></div>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        <span id="circulating-supply">0</span> / <span id="total-supply">0</span> XRP
                    </div>
                </div>
                
                <div>
                    <div class="flex justify-between text-gray-600 mb-1">
                        <span>Transaction Speed</span>
                        <span id="transaction-speed" class="text-green-600 font-medium">3-5 seconds</span>
                    </div>
                    <div class="text-xs text-gray-500 mt-1">
                        Average settlement time
                    </div>
                </div>
            </div>
        </section>

        <!-- Features Info -->
        <section class="bg-blue-50 border border-blue-200 rounded-xl p-6 mb-8 fade-in">
            <div class="flex items-start space-x-3">
                <div class="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center flex-shrink-0">
                    <svg class="w-5 h-5 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                    </svg>
                </div>
                <div>
                    <h4 class="font-semibold text-blue-800 mb-1">Fast & Efficient Digital Asset</h4>
                    <p class="text-sm text-blue-700">
                        XRP is designed for fast, low-cost international payments and remittances. Built for enterprise use with 3-5 second settlement times and minimal fees.
                    </p>
                </div>
            </div>
        </section>

        <!-- About Section -->
        <section class="mb-8 fade-in">
            <div class="flex justify-between items-center mb-4">
                <h3 class="font-semibold text-gray-800">About XRP</h3>
                <button id="toggle-description" class="text-blue-500 text-sm font-medium">Show More</button>
            </div>
            <div id="description-content" class="description-collapse text-gray-600 text-sm leading-relaxed">
                Loading XRP description...
            </div>
        </section>

        <!-- Official Links -->
        <section class="fade-in">
            <h3 class="font-semibold text-gray-800 mb-4">Resources</h3>
            <div class="grid grid-cols-2 gap-3">
                <a href="https://xrpl.org" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">XRPL.org</div>
                </a>
                <a href="https://github.com/XRPLF" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Source Code</div>
                </a>
                <a href="https://xrpl.org/xrp-ledger-overview.html" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Documentation</div>
                </a>
                <a href="blockchain_details.php" target="_blank" class="bg-gray-100 hover:bg-gray-200 rounded-lg p-3 text-center transition-colors">
                    <div class="text-blue-500 font-medium">Block Explorer</div>
                </a>
            </div>
        </section>
    </div>

    <script>
        // DOM Elements
        const xrpPriceEl = document.getElementById('xrp-price');
        const xrpArrowEl = document.getElementById('xrp-arrow');
        const xrpChangeEl = document.getElementById('xrp-change');
        const xrpEquivalentEl = document.getElementById('xrp-equivalent');
        const xrpBalanceUsdEl = document.getElementById('xrp-balance-usd');
        const marketCapEl = document.getElementById('market-cap');
        const volumeEl = document.getElementById('volume');
        const supplyEl = document.getElementById('supply');
        const marketCapRankEl = document.getElementById('market-cap-rank');
        const transactionSpeedEl = document.getElementById('transaction-speed');
        const descriptionContentEl = document.getElementById('description-content');
        const toggleDescriptionBtn = document.getElementById('toggle-description');
        const connectionStatusEl = document.getElementById('connection-status');
        const supplyProgressEl = document.getElementById('supply-progress');
        const circulatingSupplyEl = document.getElementById('circulating-supply');
        const totalSupplyEl = document.getElementById('total-supply');

        // User data from PHP - FIXED: Now using USD value correctly
        const userXRP_USD = <?php echo json_encode($xrp_usd_value); ?>;
        const initialXRPPrice = <?php echo json_encode($xrp_price); ?>;
        let previousPrice = initialXRPPrice;
        let coinData = {};
        let descriptionExpanded = false;

        // Toggle description
        toggleDescriptionBtn.addEventListener('click', () => {
            descriptionExpanded = !descriptionExpanded;
            descriptionContentEl.classList.toggle('description-collapse', !descriptionExpanded);
            descriptionContentEl.classList.toggle('description-expanded', descriptionExpanded);
            toggleDescriptionBtn.textContent = descriptionExpanded ? 'Show Less' : 'Show More';
        });

        // Format numbers with commas
        function formatNumber(num, decimals = 0) {
            return new Intl.NumberFormat('en-US', {
                minimumFractionDigits: decimals,
                maximumFractionDigits: decimals
            }).format(num);
        }

        // Format currency
        function formatCurrency(amount) {
            return new Intl.NumberFormat('en-US', {
                style: 'currency',
                currency: 'USD',
                minimumFractionDigits: 2,
                maximumFractionDigits: 6
            }).format(amount);
        }

        // Update UI with price data
        function updatePriceUI(price, change) {
            // Update price
            xrpPriceEl.textContent = formatCurrency(price);
            
            // Update 24h change
            const changeFixed = change.toFixed(2);
            xrpChangeEl.textContent = `${changeFixed}% (24h)`;
            xrpChangeEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update arrow
            xrpArrowEl.textContent = change >= 0 ? '↑' : '↓';
            xrpArrowEl.style.color = change >= 0 ? '#10B981' : '#EF4444';
            
            // Update equivalent
            const xrpEquivalent = 1 / price;
            xrpEquivalentEl.textContent = xrpEquivalent.toFixed(4) + ' XRP';
            
            // FIXED: USD balance remains constant, just display it
            xrpBalanceUsdEl.textContent = formatNumber(userXRP_USD, 2);
            
            // Price change animation
            if (previousPrice && previousPrice !== price) {
                xrpPriceEl.classList.remove('text-green-500', 'text-red-500');
                void xrpPriceEl.offsetWidth; // Trigger reflow
                xrpPriceEl.classList.add(price > previousPrice ? 'text-green-500' : 'text-red-500');
                
                setTimeout(() => {
                    xrpPriceEl.classList.remove('text-green-500', 'text-red-500');
                    xrpPriceEl.classList.add('text-gray-800');
                }, 2000);
            }
            
            previousPrice = price;
        }

        // Update market data
        function updateMarketData(data) {
            marketCapEl.textContent = formatCurrency(data.market_data.market_cap.usd);
            volumeEl.textContent = formatCurrency(data.market_data.total_volume.usd);
            supplyEl.textContent = formatNumber(data.market_data.circulating_supply) + ' XRP';
            marketCapRankEl.textContent = '#' + data.market_cap_rank;
            
            // Calculate supply percentage
            const circulating = data.market_data.circulating_supply;
            const total = data.market_data.total_supply;
            const percentage = (circulating / total) * 100;
            
            supplyProgressEl.style.width = percentage + '%';
            circulatingSupplyEl.textContent = formatNumber(circulating);
            totalSupplyEl.textContent = formatNumber(total);
        }

        // Fetch XRP data from CoinGecko
        async function fetchXRPData() {
            try {
                const response = await axios.get('https://api.coingecko.com/api/v3/coins/ripple', {
                    params: {
                        localization: false,
                        tickers: false,
                        community_data: false,
                        developer_data: false,
                        sparkline: false
                    },
                    timeout: 10000
                });
                
                coinData = response.data;
                const price = coinData.market_data.current_price.usd;
                const change = coinData.market_data.price_change_percentage_24h;
                
                updatePriceUI(price, change);
                updateMarketData(coinData);
                
                // Update description
                descriptionContentEl.innerHTML = coinData.description.en.replace(/\. /g, '.<br><br>');
                
                connectionStatusEl.style.backgroundColor = '#10B981';
                return true;
            } catch (error) {
                console.error('Error fetching XRP data:', error);
                connectionStatusEl.style.backgroundColor = '#EF4444';
                return false;
            }
        }

        // Initialize the page
        document.addEventListener('DOMContentLoaded', () => {
            // Initial data fetch
            fetchXRPData();
            
            // Refresh data every 2 minutes
            setInterval(fetchXRPData, 120000);
            
            // Animate elements on load
            document.querySelectorAll('.fade-in').forEach((el, i) => {
                el.style.animationDelay = `${i * 0.1}s`;
            });
        });
    </script>
</body>
</html>