<?php
require_once '../../db.php';
require_once '../../session.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id']) || !isset($_SESSION['current_transaction'])) {
    echo json_encode(['success' => false, 'message' => 'No active transaction']);
    exit();
}

$userId = (int)$_SESSION['user_id'];
$transid = $_SESSION['current_transaction'];

try {
    // First, check if the transaction is already complete
    $stmt = $conn->prepare("SELECT status, amount, signa FROM signalpay WHERE transid = ? AND user_id = ?");
    $stmt->bind_param("si", $transid, $userId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode(['success' => false, 'message' => 'Transaction not found']);
        exit();
    }
    
    $transaction = $result->fetch_assoc();
    
    if ($transaction['status'] !== 'complete') {
        echo json_encode(['success' => false, 'message' => 'Cannot process payment: transaction is not complete']);
        exit();
    }
    
    $conn->begin_transaction();
    
    try {
        // Update user balances based on the transaction
        // Lock all currencies for update
        $stmtUser = $conn->prepare("SELECT bnb, sol, ada, doge, btc, eth, usdt, xrp FROM users WHERE id = ? FOR UPDATE");
        $stmtUser->bind_param("i", $userId);
        $stmtUser->execute();
        $resultUser = $stmtUser->get_result();
        
        if ($resultUser->num_rows === 0) {
            throw new Exception("User not found");
        }
        
        $userBalances = array_map('floatval', $resultUser->fetch_assoc());
        $newBalances = $userBalances;
        
        // Deduct the purchase price from the total balance using a currency priority list
        $remaining = (float)$transaction['amount'];
        foreach (['bnb', 'sol', 'ada', 'doge', 'btc', 'eth', 'usdt', 'xrp'] as $currency) {
            if ($remaining <= 0) break;
            $deduct = min($userBalances[$currency], $remaining);
            $newBalances[$currency] -= $deduct;
            $remaining -= $deduct;
        }
        
        if ($remaining > 0) {
            throw new Exception("Insufficient funds for purchase");
        }
        
        // Update user balances with the new values and update the percent column
        $update = $conn->prepare("UPDATE users SET 
            bnb = ?, sol = ?, ada = ?, doge = ?,
            btc = ?, eth = ?, usdt = ?, xrp = ?,
            percent = ?
            WHERE id = ?");
        $update->bind_param("dddddddddi",
            $newBalances['bnb'], $newBalances['sol'], $newBalances['ada'], $newBalances['doge'],
            $newBalances['btc'], $newBalances['eth'], $newBalances['usdt'], $newBalances['xrp'],
            $transaction['signa'], $userId
        );
        $update->execute();
        
        // Update transaction to mark it as processed
        $updateTx = $conn->prepare("UPDATE signalpay SET processed = 1 WHERE transid = ?");
        $updateTx->bind_param("s", $transid);
        $updateTx->execute();
        
        $conn->commit();
        echo json_encode(['success' => true, 'message' => 'Payment processed successfully']);
        
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
?>