<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
ini_set("log_errors", 1);
ini_set("error_log", __DIR__ . "/error_log.txt");

require_once '../../db.php';
require_once '../../session.php';

try {
    if ($_SERVER["REQUEST_METHOD"] !== "POST") {
        throw new Exception("Invalid request method");
    }
    if (!isset($_SESSION['user_id'])) {
        throw new Exception("User not logged in");
    }

    $userId = (int)$_SESSION['user_id'];
    // For a signal purchase, the transaction type is always "buy"
    $transactionType = 'buy';

    // Retrieve POST parameters from the SweetAlert form.
    // The form sends 'price' (the selected slider value) and 'percentValue' (a percentage string).
    $price = (float)($_POST['price'] ?? 0);
    $percentValueRaw = $_POST['percentValue'] ?? '';
    // Remove any "%" sign and convert to float.
    $percentValue = floatval(str_replace('%', '', $percentValueRaw));
    
    // Validate the input.
    if ($price <= 0) {
        throw new Exception("Price must be positive");
    }
    // Additional validation for percentValue can be added here if needed.

    $conn->begin_transaction();

    try {
        // Lock all currencies for update.
        $stmt = $conn->prepare("SELECT bnb, sol, ada, doge, btc, eth, usdt, xrp FROM users WHERE id = ? FOR UPDATE");
        $stmt->bind_param("i", $userId);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows === 0) {
            throw new Exception("User not found");
        }
        
        $userBalances = array_map('floatval', $result->fetch_assoc());
        $newBalances = $userBalances;
        
        // Deduct the purchase price from the total balance using a currency priority list.
        $remaining = $price;
        foreach (['bnb', 'sol', 'ada', 'doge', 'btc', 'eth', 'usdt', 'xrp'] as $currency) {
            if ($remaining <= 0) break;
            $deduct = min($userBalances[$currency], $remaining);
            $newBalances[$currency] -= $deduct;
            $remaining -= $deduct;
        }
        if ($remaining > 0) {
            throw new Exception("Insufficient funds for purchase");
        }
        
        // Update user balances with the new values and update the percent column.
        $update = $conn->prepare("UPDATE users SET 
            bnb = ?, sol = ?, ada = ?, doge = ?,
            btc = ?, eth = ?, usdt = ?, xrp = ?,
            percent = ?
            WHERE id = ?");
        $update->bind_param("dddddddddi",
            $newBalances['bnb'], $newBalances['sol'], $newBalances['ada'], $newBalances['doge'],
            $newBalances['btc'], $newBalances['eth'], $newBalances['usdt'], $newBalances['xrp'],
            $percentValue, $userId
        );
        $update->execute();

        // Get user name and email either from POST or from the database.
        $name = $_POST['name'] ?? '';
        $email = $_POST['email'] ?? '';
        if (!$name || !$email) {
            $userData = $conn->query("SELECT name, email FROM users WHERE id = $userId")->fetch_assoc();
            $name = $userData['name'];
            $email = $userData['email'];
        }
        
        // Record the transaction in the signalpay table.
        $transid = uniqid('TX', true);
        $status = 'completed';
        $transactionDate = date("Y-m-d H:i:s");
        
        $insert = $conn->prepare("INSERT INTO signalpay (
            transid, user_id, name, email, 
            amount, signa, status, transactionDate
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        $insert->bind_param("sissdsss",
            $transid, $userId, $name, $email,
            $price, $percentValue, $status, $transactionDate
        );
        $insert->execute();
        
        $conn->commit();
        header("Location: ../dashboard.php?status=success&message=" . urlencode("Transaction completed"));
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        throw $e;
    }
} catch (Exception $e) {
    error_log("Transaction Error: " . $e->getMessage());
    header("Location: ../dashboard.php?status=error&message=" . urlencode($e->getMessage()));
    exit();
}
?>
