<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Payment Confirmation - USDT</title>
  <link rel="icon" href="https://bitnovaspro.com/cryptos/usdt.png" type="image/png">
  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    @keyframes bounce-in {
      0% {
        transform: scale(0.3);
        opacity: 0;
      }
      50% {
        transform: scale(1.05);
      }
      70% {
        transform: scale(0.9);
      }
      100% {
        transform: scale(1);
        opacity: 1;
      }
    }

    @keyframes fade-in-up {
      from {
        opacity: 0;
        transform: translateY(30px);
      }
      to {
        opacity: 1;
        transform: translateY(0);
      }
    }

    @keyframes checkmark {
      0% {
        stroke-dashoffset: 100;
      }
      100% {
        stroke-dashoffset: 0;
      }
    }

    @keyframes circle {
      0% {
        stroke-dashoffset: 200;
      }
      100% {
        stroke-dashoffset: 0;
      }
    }

    .bounce-in {
      animation: bounce-in 0.6s ease-out;
    }

    .fade-in-up {
      animation: fade-in-up 0.6s ease-out;
    }

    .fade-in-up-delay-1 {
      animation: fade-in-up 0.6s ease-out 0.2s both;
    }

    .fade-in-up-delay-2 {
      animation: fade-in-up 0.6s ease-out 0.4s both;
    }

    .fade-in-up-delay-3 {
      animation: fade-in-up 0.6s ease-out 0.6s both;
    }

    .checkmark-circle {
      stroke-dasharray: 200;
      stroke-dashoffset: 200;
      animation: circle 0.6s ease-in-out 0.8s forwards;
    }

    .checkmark-check {
      stroke-dasharray: 100;
      stroke-dashoffset: 100;
      animation: checkmark 0.3s ease-in-out 1.2s forwards;
    }

    .progress-bar {
      transition: width 1s ease-in-out;
    }
  </style>
</head>
<body class="bg-gradient-to-br from-gray-50 to-gray-100 min-h-screen">
  <?php
  session_start();

  // Validate session and transaction data
  if (
      !isset($_SESSION['transaction_data']) ||
      !isset(
          $_SESSION['transaction_data']['network'],
          $_SESSION['transaction_data']['total_amount'],
          $_SESSION['transaction_data']['wallet'],
          $_SESSION['transaction_data']['fee'],
          $_SESSION['transaction_data']['transaction_status']
      ) ||
      $_SESSION['transaction_data']['transaction_status'] !== 'completed'
  ) {
      header("Location: usdt.php");
      exit();
  }

  $transaction_data = $_SESSION['transaction_data'];

  // Validate numeric values
  if (!is_numeric($transaction_data['total_amount']) || $transaction_data['total_amount'] <= 0) {
      die("Invalid transaction amount");
  }

  // For USDT, we'll use the direct amount since it's already in USD equivalent
  $usd_amount = (float)$transaction_data['total_amount'];
  $usdt_amount = $usd_amount; // USDT is pegged to USD

  // If USDT value is available from session, use it
  if (isset($transaction_data['usdt']) && is_numeric($transaction_data['usdt'])) {
      $usdt_amount = (float)$transaction_data['usdt'];
  }

  // Formatting
  $formatted_amount = number_format($usdt_amount, 2);
  $crypto_symbol = 'USDT';
  $crypto_icon = 'https://bitnovaspro.com/cryptos/usdt.png';

  // Transaction details
  $transaction_hash = $transaction_data['transhash'] ?? 'N/A';
  $processed_at = $transaction_data['processed_at'] ?? date('Y-m-d H:i:s');
  $user_name = $transaction_data['name'] ?? 'User';
  ?>

  <div class="flex flex-col items-center justify-center min-h-screen p-4">
    <div class="bg-white rounded-2xl shadow-2xl w-full max-w-md p-8 bounce-in">
      <div class="text-center space-y-6">

        <!-- Greeting -->
        <div class="fade-in-up mb-4">
          <h1 class="text-lg font-medium text-gray-600 mb-2">Hello, <?= htmlspecialchars($user_name) ?>!</h1>
          <div class="flex items-center justify-center gap-3">
            <img
              src="<?= htmlspecialchars($crypto_icon) ?>"
              alt="<?= htmlspecialchars($crypto_symbol) ?> logo"
              class="w-10 h-10 rounded-full shadow-lg"
            >
            <span class="bg-gradient-to-r from-green-100 to-emerald-100 text-green-800 px-4 py-2 rounded-full text-sm font-semibold shadow-sm">
              Payment Completed
            </span>
          </div>
        </div>

        <!-- Amount -->
        <div class="fade-in-up-delay-1">
          <h2 class="text-2xl font-bold text-gray-800 mb-2">Successfully Sent</h2>
          <p class="text-4xl font-bold text-gray-900 mb-1">
            <?= $formatted_amount ?>
            <span class="text-2xl text-gray-600 font-medium"><?= htmlspecialchars($crypto_symbol) ?></span>
          </p>
          <p class="text-sm text-gray-500">
            ≈ $<?= number_format($usdt_amount, 2) ?> USD
          </p>
        </div>

        <!-- Animated Checkmark -->
        <div class="fade-in-up-delay-2 flex justify-center">
          <svg class="w-16 h-16" viewBox="0 0 100 100">
            <circle
              cx="50"
              cy="50"
              r="45"
              fill="none"
              stroke="#10B981"
              stroke-width="4"
              class="checkmark-circle"
            />
            <path
              d="M30 50 L45 65 L70 35"
              fill="none"
              stroke="#10B981"
              stroke-width="4"
              stroke-linecap="round"
              stroke-linejoin="round"
              class="checkmark-check"
            />
          </svg>
        </div>

        <!-- Transaction Details -->
        <div class="bg-gray-50 rounded-xl p-4 text-sm text-gray-700 space-y-3 fade-in-up-delay-3">
          <div class="flex justify-between items-center">
            <span class="font-medium">Transaction ID:</span>
            <span class="font-mono text-xs bg-white px-2 py-1 rounded border max-w-32 truncate" title="<?= htmlspecialchars($transaction_hash) ?>">
              <?= htmlspecialchars(substr($transaction_hash, 0, 8)) ?>...<?= htmlspecialchars(substr($transaction_hash, -8)) ?>
            </span>
          </div>
          <div class="flex justify-between items-center">
            <span class="font-medium">To Wallet:</span>
            <span class="font-mono text-xs bg-white px-2 py-1 rounded border" title="<?= htmlspecialchars($transaction_data['wallet']) ?>">
              <?= htmlspecialchars(substr($transaction_data['wallet'], 0, 8)) ?>...<?= htmlspecialchars(substr($transaction_data['wallet'], -8)) ?>
            </span>
          </div>
          <div class="flex justify-between items-center">
            <span class="font-medium">Network:</span>
            <span class="font-semibold capitalize">
              <?= htmlspecialchars($transaction_data['network']) ?>
            </span>
          </div>
          <div class="flex justify-between items-center">
            <span class="font-medium">Network Fee:</span>
            <span class="font-semibold">
              <?= htmlspecialchars($transaction_data['fee']) ?> <?= htmlspecialchars($crypto_symbol) ?>
            </span>
          </div>
          <div class="flex justify-between items-center">
            <span class="font-medium">Processed At:</span>
            <span class="text-xs">
              <?= date('M j, Y \a\t g:i A', strtotime($processed_at)) ?>
            </span>
          </div>
        </div>

        <!-- USDT Balance Display -->
        <?php if (isset($transaction_data['new_balance'])): ?>
        <div class="bg-green-50 rounded-xl p-4 text-sm fade-in-up-delay-3">
          <div class="flex justify-between items-center">
            <span class="font-medium text-green-800">New USDT Balance:</span>
            <span class="font-bold text-green-900">
              <?= number_format((float)$transaction_data['new_balance'], 2) ?> USDT
            </span>
          </div>
        </div>
        <?php endif; ?>

        <!-- Countdown Redirect -->
        <div class="fade-in-up-delay-3">
          <div class="bg-blue-50 rounded-xl p-4 text-center">
            <p class="text-sm text-blue-800 mb-2">
              Redirecting to dashboard in <span id="countdown" class="font-bold">30</span> seconds
            </p>
            <div class="w-full bg-blue-200 rounded-full h-2 mb-3">
              <div id="progress-bar" class="bg-blue-600 h-2 rounded-full progress-bar" style="width: 100%"></div>
            </div>
            <button
              onclick="redirectNow()"
              class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium transition-colors duration-200"
            >
              Go to Dashboard Now
            </button>
          </div>
        </div>

      </div>
    </div>
  </div>

  <!-- JS countdown and redirect logic -->
  <script>
    let countdown = 30;
    let countdownInterval;
    let progressInterval;

    function updateCountdown() {
      const countdownElement = document.getElementById('countdown');
      const progressBar = document.getElementById('progress-bar');
      countdownElement.textContent = countdown;
      const progressPercentage = (countdown / 30) * 100;
      progressBar.style.width = progressPercentage + '%';
      countdown--;
      if (countdown < 0) {
        clearInterval(countdownInterval);
        clearInterval(progressInterval);
        redirectToDashboard();
      }
    }

    function redirectToDashboard() {
      clearSessionData();
      document.body.style.opacity = '0';
      document.body.style.transition = 'opacity 0.5s ease-out';
      setTimeout(() => {
        window.location.href = "dashboard.php";
      }, 500);
    }

    function redirectNow() {
      clearInterval(countdownInterval);
      clearInterval(progressInterval);
      redirectToDashboard();
    }

    function clearSessionData() {
      fetch('clear_session.php', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({action: 'clear_transaction_data'})
      }).catch(error => {
        console.log('Session cleanup request failed:', error);
      });
    }

    window.addEventListener('load', function() {
      setTimeout(() => {
        countdownInterval = setInterval(updateCountdown, 1000);
        updateCountdown();
      }, 1000);
    });

    window.addEventListener('beforeunload', function() {
      clearSessionData();
    });

    window.addEventListener('pageshow', function(event) {
      if (event.persisted) {
        window.location.href = "dashboard.php";
      }
    });
  </script>
</body>
</html>